﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/detail/fs_Newable.h>

namespace nn { namespace fssystem {

/*!
* @brief   ファイルシステム用バディヒープクラス
*
* @details 本クラスでは最小割り当て単位をブロックと呼びます。
*          ブロックのサイズは 2^m (std::log2(BlockSizeMin) <= m) バイトでなければなりません。
*
*          連続した 2^n 個のブロックを結合したものをページと呼びます。
*          このとき、そのページのオーダーは n であるといいます。
*          オーダーは 0 以上であり、最大のオーダーは　OrderUpperLimit 未満の値を初期化時に指定します。
*/
class FileSystemBuddyHeap
{
    NN_DISALLOW_COPY(FileSystemBuddyHeap);
    NN_DISALLOW_MOVE(FileSystemBuddyHeap);

public:
    //! Initialize() に渡す領域はこの定数でアライメントされている必要があります。
    static const size_t BufferAlignment = sizeof(void*);

    //! ブロックサイズはこの定数以上の値を指定する必要があります。
    static const size_t BlockSizeMin = sizeof(void*) * 2;

    //! オーダーはこの定数未満の値を指定する必要があります。
    static const int OrderUpperLimit = sizeof(int) * 8 - 1;

public:
    /*!
    * @brief       オーダーからブロック数を求めます。
    *
    * @param[in]   order   オーダー
    *
    * @return      ページ数を返します。
    *
    * @pre
    *              - 0 <= order < OrderUpperLimit
    */
    static int GetBlockCountFromOrder(int order) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
        NN_SDK_REQUIRES_LESS(order, OrderUpperLimit);
        return (1 << order);
    }

    /*!
    * @brief       総サイズとブロックサイズから割り当て可能な最大のオーダーを取得します。
    *
    * @param[in]   size        ヒープに割り当てる領域のサイズ
    * @param[in]   blockSize   ブロックのサイズ
    *
    * @return      割り当て可能な最大のオーダーを返します。
    *              size が大きすぎる場合は負の値を返します。
    *
    * @details     本関数によって取得した値をそのまま Initialize() に渡すことができます。
    *
    * @pre
    *              - blockSize <= size
    *              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
    */
    static int QueryOrderMax(size_t size, size_t blockSize) NN_NOEXCEPT;

    /*!
    * @brief       ワークバッファに必要なサイズを計算します。
    *
    * @param[in]   orderMax    最大ページ数のオーダー
    *
    * @return      ワークバッファに必要なサイズを返します。
    *
    * @pre
    *              - 0 < orderMax < OrderUpperLimit
    */
    static size_t QueryWorkBufferSize(int orderMax) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(orderMax, 1, OrderUpperLimit);
        return nn::util::align_up(sizeof(PageList) * (orderMax + 1) + NN_ALIGNOF(PageList), 8);
    }

    /*!
    * @brief       コンストラクタです。
    */
    FileSystemBuddyHeap() NN_NOEXCEPT
    : m_BlockSize(0U),
      m_OrderMax(0),
      m_HeapStart(0U),
      m_HeapSize(0U),
      m_pFreeAreas(),
      m_TotalFreeSize(0U),
      m_pExternalFreeAreas(nullptr),
      m_pInternalFreeAreas(nullptr)
    {
    }

    /*!
    * @brief       バディヒープを初期化します。
    *
    * @param[in]   addr        ヒープに割り当てる領域のアドレス
    * @param[in]   size        ヒープに割り当てる領域のサイズ
    * @param[in]   blockSize   ブロックのサイズ
    *
    * @return      関数の実行結果を返します。
    *
    * @pre
    *              - 未初期化である
    *              - addr != nullptr
    *              - nn::util::is_aligned(addr, BufferAlignment)
    *              - blockSize <= size
    *              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
    *
    * @details     指定の領域から割り当てられる最大のオーダーが自動設定されます。
    *
    *              先頭アドレスは BufferAlignment でアライメントされている必要があります。
    *              サイズは BufferSizeMin 以上でなければなりません。
    */
    nn::Result Initialize(uintptr_t addr, size_t size, size_t blockSize) NN_NOEXCEPT
    {
        return Initialize(addr, size, blockSize, QueryOrderMax(size, blockSize));
    }

    /*!
    * @brief       バディヒープを初期化します。
    *
    * @param[in]   addr        ヒープに割り当てる領域のアドレス
    * @param[in]   size        ヒープに割り当てる領域のサイズ
    * @param[in]   blockSize   ブロックのサイズ
    * @param[in]   orderMax    最大ページ数のオーダー
    *
    * @return      関数の実行結果を返します。
    *
    * @pre
    *              - 未初期化である
    *              - addr != nullptr
    *              - nn::util::is_aligned(addr, BufferAlignment)
    *              - blockSize <= size
    *              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
    *              - 0 < orderMax < OrderUpperLimit
    *
    * @details     先頭アドレスは BufferAlignment でアライメントされている必要があります。
    *              サイズは BufferSizeMin 以上でなければなりません。
    */
    nn::Result Initialize(uintptr_t addr, size_t size, size_t blockSize, int orderMax) NN_NOEXCEPT;

    /*!
    * @brief       バディヒープを初期化します。
    *
    * @param[in]   addr            ヒープに割り当てる領域のアドレス
    * @param[in]   size            ヒープに割り当てる領域のサイズ
    * @param[in]   blockSize       ブロックのサイズ
    * @param[in]   workBuffer      ワークバッファ
    * @param[in]   workBufferSize  ワークバッファのサイズ
    *
    * @return      関数の実行結果を返します。
    *
    * @pre
    *              - 未初期化である
    *              - addr != nullptr
    *              - nn::util::is_aligned(addr, BufferAlignment)
    *              - blockSize <= size
    *              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
    *              - workBuffer != nullptr
    *              - QueryWorkBufferSize(QueryOrderMax(size, blockSize)) <= workBufferSize
    *
    * @details     ワークバッファを外部から渡します。
    *
    *              指定の領域から割り当てられる最大のオーダーが自動設定されます。
    *
    *              先頭アドレスは BufferAlignment でアライメントされている必要があります。
    *              サイズは BufferSizeMin 以上でなければなりません。
    */
    nn::Result Initialize(
                   uintptr_t addr,
                   size_t size,
                   size_t blockSize,
                   void* workBuffer,
                   size_t workBufferSize
               ) NN_NOEXCEPT
    {
        return Initialize(addr, size, blockSize, QueryOrderMax(size, blockSize), workBuffer, workBufferSize);
    }

    /*!
    * @brief       バディヒープを初期化します。
    *
    * @param[in]   addr            ヒープに割り当てる領域のアドレス
    * @param[in]   size            ヒープに割り当てる領域のサイズ
    * @param[in]   blockSize       ブロックのサイズ
    * @param[in]   orderMax        最大ページ数のオーダー
    * @param[in]   workBuffer      ワークバッファ
    * @param[in]   workBufferSize  ワークバッファのサイズ
    *
    * @return      関数の実行結果を返します。
    *
    * @pre
    *              - 未初期化である
    *              - addr != nullptr
    *              - nn::util::is_aligned(addr, BufferAlignment)
    *              - blockSize <= size
    *              - blockSize = 2^m (std::log2(BlockSizeMin) <= m)
    *              - 0 < orderMax < OrderUpperLimit
    *              - workBuffer != nullptr
    *              - QueryWorkBufferSize(orderMax) <= workBufferSize
    *
    * @details     ワークバッファを外部から渡します。
    *
    *              先頭アドレスは BufferAlignment でアライメントされている必要があります。
    *              サイズは BufferSizeMin 以上でなければなりません。
    */
    nn::Result Initialize(
                   uintptr_t addr,
                   size_t size,
                   size_t blockSize,
                   int orderMax,
                   void* workBuffer,
                   size_t workBufferSize
               ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, QueryWorkBufferSize(orderMax));
        NN_UNUSED(workBufferSize);
        const auto alignedBuffer
            = nn::util::align_up(reinterpret_cast<uintptr_t>(workBuffer), NN_ALIGNOF(PageList));
        m_pExternalFreeAreas = reinterpret_cast<PageList*>(alignedBuffer);
        return Initialize(addr, size, blockSize, orderMax);
    }

    /*!
    * @brief       バディヒープを破棄します。
    *
    * @details     割り当てられたまま解放されていないページがあったとしても何もしません。
    *
    * @pre
    *              - 初期化済みである
    */
    void Finalize() NN_NOEXCEPT;

    /*!
    * @brief       オーダーを指定してページを確保します。
    *
    * @param[in]   order   確保するページのオーダー
    *
    * @return      確保したページへのポインタ
    * @retval      nullptr 空き領域が不足しています。
    *
    * @pre
    *              - 初期化済みである
    *              - 0 <= order <= GetOrderMax()
    */
    void* AllocateByOrder(int order) NN_NOEXCEPT;

    /*!
    * @brief       メモリを解放します。
    *
    * @param[in]   ptr     解放するページへのポインタ
    * @param[in]   order   解放するページのオーダー
    *
    * @pre
    *              - 初期化済みである
    *              - ptr がこのヒープによって確保された未解放の領域を指しているか nullptr である
    *              - 0 <= order <= GetOrderMax()
    *
    * @details     ptr に nullptr を指定すると何もしません。
    */
    void Free(void* ptr, int order) NN_NOEXCEPT;

    /*!
    * @brief       指定のサイズ以上で最小のページのオーダーを取得します。
    *
    * @param[in]   size    バイト単位のサイズ
    *
    * @return      指定のサイズ以上で最小のページのオーダーを返します。
    *
    * @pre
    *              - 初期化済みである
    */
    int GetOrderFromBytes(size_t size) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
        return GetOrderFromBlockCount(GetBlockCountFromSize(size));
    }

    /*!
    * @brief       指定のオーダーを持つページのバイト単位のサイズを計算します。
    *
    * @param[in]   order   オーダー
    *
    * @return      バイト単位のサイズ
    *
    * @pre
    *              - 初期化済みである
    *              - 0 <= order <= GetOrderMax()
    */
    size_t GetBytesFromOrder(int order) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
        NN_SDK_REQUIRES_GREATER_EQUAL(order, 0);
        NN_SDK_REQUIRES_LESS_EQUAL(order, GetOrderMax());
        return (GetBlockSize() << order);
    }

    /*!
    * @brief       このヒープから確保できるページのオーダーの最大を取得します。
    *
    * @return      このヒープから確保できるページのオーダーの最大を返します。
    *
    * @pre
    *              - 初期化済みである
    */
    int GetOrderMax() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
        return m_OrderMax;
    }

    /*!
    * @brief       ブロックサイズを取得します。
    *
    * @return      ブロックサイズを返します。
    *
    * @pre
    *              - 初期化済みである
    */
    size_t GetBlockSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
        return m_BlockSize;
    }

    /*!
    * @brief       ページあたりのブロック数の最大を取得します。
    *
    * @return      ページあたりのブロック数の最大を返します。
    *
    * @pre
    *              - 初期化済みである
    */
    int GetPageBlockCountMax() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
        return 1 << GetOrderMax();
    }

    /*!
    * @brief       ページサイズの最大をバイト単位で取得します。
    *
    * @return      ページサイズの最大を返します。
    *
    * @pre
    *              - 初期化済みである
    */
    size_t GetPageSizeMax() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pFreeAreas);
        return GetPageBlockCountMax() * GetBlockSize();
    }

    /*!
    * @brief       ヒープの空き容量の合計を取得します。
    *
    * @return      ヒープの空き容量の合計のバイト数
    *
    * @pre
    *              - 初期化済みである
    */
    size_t GetTotalFreeSize() const NN_NOEXCEPT;

    /*!
    * @brief       ヒープから確保可能なページサイズの最大を取得します。
    *
    * @return      ヒープから確保可能なページサイズの最大のバイト数
    *
    * @pre
    *              - 初期化済みである
    */
    size_t GetAllocatableSizeMax() const NN_NOEXCEPT;

    /*!
    * @brief       ヒープの管理状態をログに出力します。
    *
    * @details     デバッグ用の関数です。
    *
    * @pre
    *              - 初期化済みである
    */
    void Dump() const NN_NOEXCEPT;

private:
    class PageList;

    //! ページエントリーです。
    struct PageEntry
    {
        PageEntry* pNext;  //!< 次のページのエントリーです。
    };

    NN_STATIC_ASSERT(sizeof(PageEntry) <= BlockSizeMin);
    NN_STATIC_ASSERT(std::is_pod<PageEntry>::value);

    //! 同一サイズのメモリブロック毎にリンクリストを構成します。
    class PageList : public nn::fs::detail::Newable
    {
        NN_DISALLOW_COPY(PageList);
        NN_DISALLOW_MOVE(PageList);

    public:
        /*!
        * @brief       コンストラクタです。
        */
        PageList() NN_NOEXCEPT
        : m_pFirstPageEntry(nullptr),
          m_pLastPageEntry(nullptr),
          m_EntryCount(0)
        {
        }

        /*!
        * @brief       ページリストが空かどうか判定します。
        *
        * @return      判定結果を返します。
        * @retval      true    ページリストには何も登録されていません。
        * @retval      false   ページリストにはエントリが登録されています。
        */
        bool IsEmpty() const NN_NOEXCEPT
        {
            return m_EntryCount == 0;
        }

        /*!
        * @brief       要素を取り除かずにページリストの先頭から 1 ページ取得します。
        *
        * @return      ページリストの先頭要素。
        * @retval      nullptr ページリストが空です。
        */
        const PageEntry* GetFront() const NN_NOEXCEPT
        {
            return m_pFirstPageEntry;
        }

        /*!
        * @brief       ページリストの先頭から 1 ページ取得します。
        *
        * @return      ページリストの先頭要素。
        *
        * @pre
        *              - リストに少なくとも 1 つのエントリが登録されている
        */
        PageEntry* PopFront() NN_NOEXCEPT;

        /*!
        * @brief       ページリストの最後尾に 1 ページ追加します。
        *
        * @param[in]   pPageEntry  追加対象のページ
        *
        * @pre
        *              - pPageEntry != nullptr
        *              - pPageEntry がどのリストにも登録されていない
        */
        void PushBack(PageEntry* pPageEntry) NN_NOEXCEPT;

        /*!
        * @brief       ページリストから 1 ページ削除します。
        *
        * @param[in]   pPageEntry  削除対象のページエントリ
        *
        * @return      削除に成功したかどうかを返します。
        * @retval      true        正常に削除できた。
        * @retval      false       ページが存在しない。
        *
        * @pre
        *              - pPageEntry != nullptr
        */
        bool Remove(PageEntry* pPageEntry) NN_NOEXCEPT;

        /*!
        * @brief       ページリストに含まれるページ数を取得します。
        *
        * @return      ページリストに含まれるページ数を返します。
        */
        int GetSize() const NN_NOEXCEPT
        {
            return m_EntryCount;
        }

    private:
        PageEntry* m_pFirstPageEntry; //!< ページリストの先頭要素
        PageEntry* m_pLastPageEntry;  //!< ページリストの最後尾要素
        int m_EntryCount;             //!< ページリストに登録されているエントリ数
    };

private:
    /*!
    * @brief       大きなページから必要な分だけを切り出します。
    *
    * @param[in]   pPageEntry      分割対象のページエントリー
    * @param[in]   demandedOrder   必要なオーダー
    * @param[in]   wholeOrder      分割対象全体のオーダー
    *
    * @pre
    *              - pPageEntry != nullptr
    *              - 0 <= demandedOrder <= wholeOrder <= GetOrderMax()
    */
    void DivideBuddies(PageEntry* pPageEntry, int demandedOrder, int wholeOrder) NN_NOEXCEPT;

    /*!
    * @brief       空きページリストへ登録します。あわせてメモリの連結処理を行います。
    *
    * @param[in]   pPageEntry  追加対象のページエントリー
    * @param[in]   order       オーダー
    *
    * @pre
    *              - pPageEntry != nullptr
    *              - 0 <= order <= GetOrderMax()
    */
    void JoinBuddies(PageEntry* pPageEntry, int order) NN_NOEXCEPT;

    /*!
    * @brief       指定したオーダーで指定したページとバディ関係にあるページエントリーを取得します。
    *
    * @param[in]   pPageEntry  ページエントリー
    * @param[in]   order       オーダー
    *
    * @return      指定したオーダーで pPageEntry とバディ関係にあるページエントリーを返します。
    * @retval      nullptr     バディが見つかりませんでした。
    *
    * @pre
    *              - pPageEntry != nullptr
    *              - 0 <= order <= GetOrderMax()
    */
    PageEntry* GetBuddy(PageEntry* pPageEntry, int order) NN_NOEXCEPT;

    /*!
    * @brief       指定のオーダーを持つ未割当ページのエントリーを取得します。
    *
    * @param[in]   order   オーダー
    *
    * @return      ページエントリー
    *
    * @pre
    *              - 0 <= order <= GetOrderMax()
    */
    PageEntry* GetFreePageEntry(int order) NN_NOEXCEPT;

    /*!
    * @brief       指定した値以上のブロック数からなるページのオーダーの最小を取得します。
    *
    * @param[in]   blockCount  ブロック数
    *
    * @return      指定ブロック数以上を持つページのオーダーの最小を返します。
    *              最大オーダーを超える場合は負の値を返します。
    *
    * @pre
    *              - 0 <= blockCount
    */
    int GetOrderFromBlockCount(int blockCount) const NN_NOEXCEPT;

    /*!
    * @brief       指定のサイズを格納するために必要なブロック数を取得します。
    *
    * @param[in]   size    ブロック数を取得するサイズ
    *
    * @return      指定のサイズを格納するために必要なブロック数を返します。
    */
    int GetBlockCountFromSize(size_t size) const NN_NOEXCEPT
    {
        return static_cast<int>((size + (GetBlockSize() - 1)) / GetBlockSize());
    }

    /*!
    * @brief       ページエントリーが示すアドレスを取得します。
    *
    * @param[in]   pageEntry   ページエントリー
    *
    * @return      アドレス
    *
    * @pre
    *              - m_HeapStart <= &pageEntry < m_HeapStart + m_HeapSize
    *              - std::is_aligned(&pageEntry - m_HeapStart, GetBlockSize())
    */
    uintptr_t GetAddressFromPageEntry(const PageEntry& pageEntry) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(reinterpret_cast<uintptr_t>(&pageEntry), m_HeapStart);
        NN_SDK_REQUIRES_LESS(reinterpret_cast<uintptr_t>(&pageEntry), m_HeapStart + m_HeapSize);
        NN_SDK_REQUIRES_ALIGNED(reinterpret_cast<uintptr_t>(&pageEntry) - m_HeapStart, GetBlockSize());
        return reinterpret_cast<uintptr_t>(&pageEntry);
    }

    /*!
    * @brief       指定したアドレスを含むページエントリーを取得します。
    *
    * @param[in]   addr    アドレス
    *
    * @return      ページエントリー
    *
    * @pre
    *              - m_HeapStart <= addr < m_HeapStart + m_HeapSize
    */
    PageEntry* GetPageEntryFromAddress(uintptr_t addr) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(addr, m_HeapStart);
        NN_SDK_REQUIRES_LESS(addr, m_HeapStart + m_HeapSize);
        return reinterpret_cast<PageEntry*>(((addr - m_HeapStart) & ~(GetBlockSize() - 1)) + m_HeapStart);
    }

    /*!
    * @brief       指定したページエントリーが先頭から何番目のブロックかを取得します。
    *
    * @param[in]   pageEntry   ページエントリー
    *
    * @return      先頭から何番目か
    *
    * @pre
    *              - m_HeapStart <= &pageEntry < m_HeapStart + m_HeapSize
    *              - std::is_aligned(&pageEntry - m_HeapStart, GetBlockSize())
    */
    int GetIndexFromPageEntry(const PageEntry& pageEntry) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(reinterpret_cast<uintptr_t>(&pageEntry), m_HeapStart);
        NN_SDK_REQUIRES_LESS(reinterpret_cast<uintptr_t>(&pageEntry), m_HeapStart + m_HeapSize);
        NN_SDK_REQUIRES_ALIGNED(reinterpret_cast<uintptr_t>(&pageEntry) - m_HeapStart, GetBlockSize());
        return static_cast<int>((reinterpret_cast<uintptr_t>(&pageEntry) - m_HeapStart) / GetBlockSize());
    }

    /*!
    * @brief       エントリーが指定のオーダーを持つページの先頭ブロックに配置されているか判定します。
    *
    * @param[in]   pPageEntry  判定するページエントリー
    * @param[in]   order       オーダー
    *
    * @return      結果を返します。
    * @retval      true        エントリーは先頭ブロックに配置されています。
    * @retval      false       エントリーは先頭ブロック以外に配置されています。
    */
    bool IsAlignedToOrder(const PageEntry* pPageEntry, int order) const NN_NOEXCEPT
    {
        return (GetIndexFromPageEntry(*pPageEntry) & (GetBlockCountFromOrder(order) - 1)) == 0;
    }

private:
    size_t m_BlockSize; //!< ブロックサイズ
    int    m_OrderMax;  //!< オーダー

    uintptr_t m_HeapStart; //!< ヒープに割り当てられている領域のアドレス
    size_t    m_HeapSize;  //!< ヒープに割り当てられている領域のサイズ

    PageList* m_pFreeAreas; //!< フリーリスト
    size_t    m_TotalFreeSize;  //!< フリーリストに繋がった領域の合計サイズ
    PageList* m_pExternalFreeAreas; //!< フリーリスト用外部バッファ
    std::unique_ptr<PageList[]> m_pInternalFreeAreas; //!< フリーリスト用内部バッファ
};

}}

