﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace fssrv {

class SaveDataOpenTypeSetFileStorage : public nn::fs::FileStorageBasedFileSystem
{
    NN_DISALLOW_COPY(SaveDataOpenTypeSetFileStorage);

public:
    enum class OpenType
    {
        None,
        Normal,
        Internal,
        Max
    };

public:
    SaveDataOpenTypeSetFileStorage() NN_NOEXCEPT
         : m_IsOpenedAsNormal(false),
           m_IsOpenedAsInternal(false),
           m_ActiveType(OpenType::None),
           m_SaveDataId(0)
    {
    }

    Result Initialize(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* path, nn::fs::OpenMode mode, nn::fs::SaveDataId saveDataId, OpenType type) NN_NOEXCEPT;
    Result SetOpenType(OpenType type) NN_NOEXCEPT;
    void UnsetOpenType(OpenType type) NN_NOEXCEPT;
    void SetActiveType(OpenType type) NN_NOEXCEPT
    {
        m_ActiveType = type;
    }
    const OpenType GetActiveType() const NN_NOEXCEPT
    {
        return m_ActiveType;
    }
    bool IsActive() const NN_NOEXCEPT
    {
        return m_ActiveType == OpenType::Normal || m_ActiveType == OpenType::Internal;
    }

private:
    bool m_IsOpenedAsNormal;
    bool m_IsOpenedAsInternal;
    OpenType m_ActiveType;
    nn::fs::SaveDataId m_SaveDataId;
};

class SaveDataSharedFileStorage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(SaveDataSharedFileStorage);

public:
    SaveDataSharedFileStorage() NN_NOEXCEPT
      : m_pBaseFileStorage(nullptr),
        m_SaveDataId(0),
        m_OpenType(SaveDataOpenTypeSetFileStorage::OpenType::None)
    {
    }
    virtual ~SaveDataSharedFileStorage() NN_NOEXCEPT NN_OVERRIDE;
    void Initialize(std::shared_ptr<SaveDataOpenTypeSetFileStorage> pBaseFileSystem, nn::fs::SaveDataId saveDataId, SaveDataOpenTypeSetFileStorage::OpenType type) NN_NOEXCEPT;
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OperateRange(void* outBuffer, size_t outBufferSize, fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;
    const nn::fs::SaveDataId GetSaveDataId() const NN_NOEXCEPT
    {
        return m_SaveDataId;
    }

private:
    Result AccessCheck() NN_NOEXCEPT
    {
        if( m_pBaseFileStorage->GetActiveType() == m_OpenType )
        {
            NN_RESULT_SUCCESS;
        }
        else if( m_OpenType == SaveDataOpenTypeSetFileStorage::OpenType::Normal )
        {
            // アプリからのアクセスがあった場合はアプリに切り替え
            // アプリ -> インターナルの切り替えはしない
            m_pBaseFileStorage->SetActiveType(SaveDataOpenTypeSetFileStorage::OpenType::Normal);
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW(nn::fs::ResultTargetLocked());
    }

private:
    std::shared_ptr<SaveDataOpenTypeSetFileStorage> m_pBaseFileStorage;
    nn::fs::SaveDataId m_SaveDataId;
    SaveDataOpenTypeSetFileStorage::OpenType m_OpenType;
};

class SaveDataFileStorageHolder : public nn::fs::detail::Newable
{
public:
    SaveDataFileStorageHolder() NN_NOEXCEPT
    {
    }
    ~SaveDataFileStorageHolder() NN_NOEXCEPT;
    nn::Result OpenSaveDataStorage(std::shared_ptr<fs::IStorage>* pOutStorage, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem,
        nn::fs::SaveDataId saveDataId, nn::fs::SaveDataType saveDataType, nn::fs::OpenMode mode, SaveDataOpenTypeSetFileStorage::OpenType type) NN_NOEXCEPT;
    Result Register(std::shared_ptr<SaveDataOpenTypeSetFileStorage> fileStorage, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;
    std::shared_ptr<SaveDataOpenTypeSetFileStorage> GetStorage(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;
    void Unregister(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT;

private:
    class Entry : public nn::util::IntrusiveListBaseNode<Entry>, public nn::fs::detail::Newable
    {
    public:
        Entry(std::shared_ptr<SaveDataOpenTypeSetFileStorage>&& pFileStorage, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
            : m_pSaveDataFileStorage(std::move(pFileStorage)),
              m_SaveDataId(saveDataId)
        {
        }
        const nn::fs::SaveDataId GetSaveDataId() const NN_NOEXCEPT
        {
            return m_SaveDataId;
        }
        std::shared_ptr<SaveDataOpenTypeSetFileStorage> GetStorage() NN_NOEXCEPT
        {
            return m_pSaveDataFileStorage;
        }

    private:
        std::shared_ptr<SaveDataOpenTypeSetFileStorage> m_pSaveDataFileStorage;
        nn::fs::SaveDataId m_SaveDataId;
    };


private:
    nn::util::IntrusiveList<Entry, nn::util::IntrusiveListBaseNodeTraits<Entry>> m_FileStorageList;
};

nn::Result OpenSaveDataStorage(std::shared_ptr<fs::IStorage>* pOutStorage, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem,
    nn::fs::SaveDataId saveDataId, nn::fs::SaveDataType saveDataType, nn::fs::OpenMode mode, SaveDataOpenTypeSetFileStorage::OpenType type) NN_NOEXCEPT;

}}
