﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fssrv/fssrv_SaveDataInfoReaderImpl.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace fssrv {

    class SaveDataInfoFilter
    {
    public:
        SaveDataInfoFilter() NN_NOEXCEPT : m_SpaceId(util::nullopt), m_ApplicationId(util::nullopt), m_SaveDataType(util::nullopt), m_UserId(util::nullopt), m_StaticSaveDataId(util::nullopt), m_Index(util::nullopt) {}

        SaveDataInfoFilter(util::optional<nn::fs::SaveDataSpaceId> spaceId, util::optional<nn::Bit64> applicationId, util::optional<nn::fs::SaveDataType> saveDataType, util::optional<nn::fs::UserId> userId, util::optional<nn::fs::StaticSaveDataId> staticSaveDataId, util::optional<uint16_t> index, int option) NN_NOEXCEPT
            : m_SpaceId(spaceId), m_ApplicationId(applicationId), m_SaveDataType(saveDataType), m_UserId(userId), m_StaticSaveDataId(staticSaveDataId), m_Index(index), m_EnumerateOption(option){}

        SaveDataInfoFilter(fs::SaveDataSpaceId spaceId, const fs::SaveDataFilter& filter) NN_NOEXCEPT
        {
            m_SpaceId.emplace(spaceId);
            m_EnumerateOption = filter.option;

            if (filter.filterApplicationId)
            {
                m_ApplicationId.emplace(filter.attribute.programId.value);
            }
            if (filter.filterSaveDataType)
            {
                m_SaveDataType.emplace(filter.attribute.type);
            }
            if (filter.filterSaveDataUserId)
            {
                m_UserId.emplace(filter.attribute.userId);
            }
            if (filter.filterStaticSaveDataId)
            {
                m_StaticSaveDataId.emplace(filter.attribute.staticSaveDataId);
            }
            if (filter.filterIndex)
            {
                m_Index.emplace(filter.attribute.index);
            }
        }

        bool Includes(const nn::fs::SaveDataInfo& info)
        {
            if (m_SpaceId && (info.saveDataSpaceId != m_SpaceId.value()))
            {
                return false;
            }
            if (m_ApplicationId && (info.applicationId.value != m_ApplicationId.value()))
            {
                return false;
            }
            if (m_SaveDataType && (info.saveDataType != m_SaveDataType.value()))
            {
                return false;
            }
            if (m_UserId && (info.saveDataUserId != m_UserId.value()))
            {
                return false;
            }
            if (m_StaticSaveDataId && (info.systemSaveDataId != m_StaticSaveDataId.value()))
            {
                return false;
            }
            if (m_Index && (info.index != m_Index.value()))
            {
                return false;
            }
            if (!(m_EnumerateOption & fs::SaveDataEnumerateOption_AllRank) && info.rank != fs::SaveDataRank::Primary)
            {
                return false;
            }
        return true;
    }

    private:
        util::optional<nn::fs::SaveDataSpaceId>  m_SpaceId;
        util::optional<nn::Bit64>                m_ApplicationId;
        util::optional<nn::fs::SaveDataType>     m_SaveDataType;
        util::optional<nn::fs::UserId>           m_UserId;
        util::optional<nn::fs::StaticSaveDataId> m_StaticSaveDataId;
        util::optional<uint16_t>                 m_Index;
        int                                      m_EnumerateOption = 0;
    };

    class SaveDataInfoFilterReader : public SaveDataInfoReaderImpl, public nn::fs::detail::Newable
    {
    public:
        SaveDataInfoFilterReader(std::shared_ptr<SaveDataInfoReaderImpl> reader, const SaveDataInfoFilter& filter) NN_NOEXCEPT : m_Reader(std::move(reader)), m_Filter(filter) {}
        virtual nn::Result Read(nn::sf::Out<std::int64_t> outValue, const nn::sf::OutBuffer& outEntries) NN_NOEXCEPT NN_OVERRIDE;
        virtual ~SaveDataInfoFilterReader() NN_NOEXCEPT NN_OVERRIDE {}

    private:
        std::shared_ptr<SaveDataInfoReaderImpl> m_Reader;
        SaveDataInfoFilter m_Filter;
    };

}}
