﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Directory.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/htc.h>
#endif

#include <nn/fssystem/fs_Assert.h>

namespace nn { namespace fssrv { namespace detail {

    const int MountableBisPartitionCount = 4;

    class BisPartitionRootPathForHost
    {
    public:
        BisPartitionRootPathForHost() NN_NOEXCEPT
        : m_BisRootPathMutex(false)
        {
            memset(m_BisRootPath, 0x00, sizeof(m_BisRootPath));
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            memset(m_BisRootPathByTenv, 0x00, sizeof(m_BisRootPathByTenv));
#endif
        }

        Result Set(nn::fs::BisPartitionId id, const char* rootPath) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES(rootPath != nullptr, nn::fs::ResultNullptrArgument());
            NN_FSP_REQUIRES(IsValid(id), nn::fs::ResultInvalidArgument());

            std::lock_guard<os::Mutex> scopedLock(m_BisRootPathMutex);
            if(rootPath[0] == '\0')
            {
                m_BisRootPath[GetIndex(id)][0] = '\0';
                NN_RESULT_SUCCESS;
            }

            auto length = strnlen(rootPath, nn::fs::EntryNameLengthMax);
            NN_FSP_REQUIRES(length <= nn::fs::EntryNameLengthMax, nn::fs::ResultTooLongPath());
            NN_FSP_REQUIRES(length > 0, nn::fs::ResultInvalidPath());
            NN_FSP_REQUIRES(rootPath[length - 1] == '/', nn::fs::ResultInvalidPath());

            strncpy(m_BisRootPath[GetIndex(id)], rootPath, sizeof(m_BisRootPath[0]) - 1);
            NN_RESULT_SUCCESS;
        }

        const char* Get(nn::fs::BisPartitionId id) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> scopedLock(m_BisRootPathMutex);
#if defined(NN_BUILD_CONFIG_OS_WIN32) && defined(NN_BUILD_CONFIG_HTC_ENABLED)
            char variable[64];
            switch (id)
            {
            case nn::fs::BisPartitionId::CalibrationFile:
                util::SNPrintf(variable, sizeof(variable), "SDK_FS_WIN_BIS_CALIB_ROOT_PATH"); break;
            case nn::fs::BisPartitionId::SafeMode:
                util::SNPrintf(variable, sizeof(variable), "SDK_FS_WIN_BIS_SAFE_ROOT_PATH"); break;
            case nn::fs::BisPartitionId::System:
                util::SNPrintf(variable, sizeof(variable), "SDK_FS_WIN_BIS_SYSTEM_ROOT_PATH"); break;
            case nn::fs::BisPartitionId::User:
                util::SNPrintf(variable, sizeof(variable), "SDK_FS_WIN_BIS_USER_ROOT_PATH"); break;
            default:
                NN_UNEXPECTED_DEFAULT;
            };
            size_t outLength = 0;
            if (nn::htc::GetTargetEnvironmentVariableLength(&outLength, variable).IsSuccess() && outLength <= nn::fs::EntryNameLengthMax + 1)
            {
                if (nn::htc::GetTargetEnvironmentVariable(&outLength, m_BisRootPathByTenv[GetIndex(id)], outLength, variable).IsSuccess())
                {
                    return m_BisRootPathByTenv[GetIndex(id)];
                }
            }
#endif // defined(NN_BUILD_CONFIG_OS_WIN32) && defined(NN_BUILD_CONFIG_HTC_ENABLED)
            return m_BisRootPath[GetIndex(id)];
        }

        bool IsEmulated(nn::fs::BisPartitionId id) NN_NOEXCEPT
        {
            if(strncmp(Get(id), "\0", 1) != 0)
            {
                return true;
            }
            return false;
        }

        static bool IsValid(nn::fs::BisPartitionId id) NN_NOEXCEPT
        {
            switch (id)
            {
            case nn::fs::BisPartitionId::CalibrationFile:
            case nn::fs::BisPartitionId::SafeMode:
            case nn::fs::BisPartitionId::System:
            case nn::fs::BisPartitionId::SystemProperPartition:
            case nn::fs::BisPartitionId::User:
                return true;
            default:
                return false;
            }
        }

    private:
        int GetIndex(nn::fs::BisPartitionId id) NN_NOEXCEPT
        {
            switch (id)
            {
            case nn::fs::BisPartitionId::CalibrationFile:
                return 0;
            case nn::fs::BisPartitionId::SafeMode:
                return 1;
            case nn::fs::BisPartitionId::System:
            case nn::fs::BisPartitionId::SystemProperPartition: // TORIAEZU
                return 2;
            case nn::fs::BisPartitionId::User:
                return 3;
            default:
                NN_UNEXPECTED_DEFAULT;
            };
        }

        char m_BisRootPath[MountableBisPartitionCount][nn::fs::EntryNameLengthMax + 1];
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        char m_BisRootPathByTenv[MountableBisPartitionCount][nn::fs::EntryNameLengthMax + 1];
#endif
        os::Mutex m_BisRootPathMutex;
    };

}}}
