﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>

namespace nn { namespace fs { namespace fsa {

    /**
    * @brief    ディレクトリ操作のためのインターフェースです。
    *
    * @details  全ての関数同士の呼び出しについて、複数のスレッドから同時に呼び出された場合の動作は未定義でかまいません。
    */
    class IDirectory
    {
    public:

        Result Read(int64_t* outValue, DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES( outValue   != nullptr, nn::fs::ResultNullptrArgument() );
            if( count == 0 )
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }
            NN_FSP_REQUIRES( outEntries != nullptr, nn::fs::ResultNullptrArgument() );
            NN_FSP_REQUIRES( count > 0, nn::fs::ResultInvalidArgument());
            return DoRead(outValue, outEntries, count);
        }

        Result GetEntryCount(int64_t* outValue) NN_NOEXCEPT
        {
            NN_FSP_REQUIRES( outValue   != nullptr, nn::fs::ResultNullptrArgument() );
            return DoGetEntryCount(outValue);
        }

    public:
        /**
        * @brief    nn::fs::CloseFile に応じて呼び出されるデストラクタです。
        *
        * @details  実装クラスはディレクトリのクローズ処理をデストラクタで実装してください。
        */
        virtual ~IDirectory() NN_NOEXCEPT{}

    private:
        /**
        * @brief    nn::fs::ReadDirectory に応じて呼び出されるディレクトリエントリ読み込み関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリエントリの読み込み処理を実装してください。
        *           nn::fs::ReadDirectory の DirectoryHandle 以外の引数が委譲されます。
        */
        virtual Result DoRead(int64_t* outValue, DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT = 0;

        /**
        * @brief    nn::fs::GetDirectoryCount に応じて呼び出されるディレクトリ数取得関数です。
        *
        * @details  実装クラスはこの関数の実装でディレクトリ数取得処理を実装してください。
        *           nn::fs::GetDirectoryCount の DirectoryHandle 以外の引数が委譲されます。
        */
        virtual Result DoGetEntryCount(int64_t* outValue) NN_NOEXCEPT = 0;
    };

}}}
