﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  システムセーブデータ関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace fs {

//! @name システムセーブデータ関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      ユーザーを指定してシステムセーブデータを作成します。
*
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  userId              ユーザー識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, UserId userId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      所有者とユーザーを指定してシステムセーブデータを作成します。
*
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  userId              ユーザー識別子
*   @param[in]  saveDataOwnerId     セーブデータのオーナ ID
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      所有者とユーザー、配置場所を指定してシステムセーブデータを作成します。
*
*   @param[in]  saveDataSpaceId     セーブデータの配置場所の識別子
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  userId              ユーザー識別子
*   @param[in]  saveDataOwnerId     セーブデータのオーナ ID
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSystemSaveData(SaveDataSpaceId saveDataSpaceId, SystemSaveDataId systemSaveDataId, UserId userId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ID とユーザーを指定してシステムセーブデータをマウントします。
*
*   @param[in]  name       マウント名
*   @param[in]  id         システムセーブデータの識別子
*   @param[in]  userId     ユーザー識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*/
Result MountSystemSaveData(const char* name, SystemSaveDataId id, UserId userId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ID とユーザー、配置場所を指定してシステムセーブデータをマウントします。
*
*   @param[in]  name                マウント名
*   @param[in]  id                  システムセーブデータの識別子
*   @param[in]  userId              ユーザー識別子
*   @param[in]  saveDataSpaceId     セーブデータの配置場所の識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*/
Result MountSystemSaveData(const char* name, SaveDataSpaceId saveDataSpaceId, SystemSaveDataId id, UserId userId) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ID とユーザー、配置場所を指定してシステムセーブデータを削除します。
*
*   @param[in]  saveDataSpaceId     セーブデータの配置場所の識別子
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  userId              ユーザー識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*/
Result DeleteSystemSaveData(SaveDataSpaceId saveDataSpaceId, SystemSaveDataId systemSaveDataId, const UserId& userId) NN_NOEXCEPT;

//! @}

}} // namespace nn::fs

