﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  システムセーブデータ関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SaveDataTransaction.h>
#include <nn/fs/fs_SaveDataTypes.h>

namespace nn { namespace fs {

/**
* @brief システムセーブデータの識別子です。
*/
typedef uint64_t SystemSaveDataId;

//! @name システムセーブデータ関連 API
//! @{

//--------------------------------------------------------------------------
/**
*   @brief      システムセーブデータを作成します。
*
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;



//--------------------------------------------------------------------------
/**
*   @brief      所有者を指定してシステムセーブデータを作成します。
*
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/
Result CreateSystemSaveData(SystemSaveDataId systemSaveDataId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      所有者及び配置場所を指定してシステムセーブデータを作成します。
*
*   @param[in]  saveDataSpaceId     セーブデータの配置場所の識別子
*   @param[in]  systemSaveDataId    システムセーブデータの識別子
*   @param[in]  saveDataOwnerId     所有者の識別子
*   @param[in]  size                セーブデータのデータ保存領域のバイトサイズ
*   @param[in]  journalSize         セーブデータのジャーナリング領域のバイトサイズ
*   @param[in]  flags               セーブデータのフラグ集合
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - size >= 0
*       - journalSize >= 0
*/

Result CreateSystemSaveData(SaveDataSpaceId saveDataSpaceId, SystemSaveDataId systemSaveDataId, uint64_t saveDataOwnerId, int64_t size, int64_t journalSize, uint32_t flags) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ID に紐づいたシステムセーブデータをマウントします。
*
*   @param[in]  name       マウント名
*   @param[in]  id         システムセーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*/
Result MountSystemSaveData(const char* name, SystemSaveDataId id) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      ID 及び配置場所に紐づいたシステムセーブデータをマウントします。
*
*   @param[in]  name                    マウント名
*   @param[in]  saveDataSpaceId         セーブデータの配置場所の識別子
*   @param[in]  id                      システムセーブデータの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*/
Result MountSystemSaveData(const char* name, SaveDataSpaceId saveDataSpaceId, SystemSaveDataId id) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
*   @brief      セーブデータのマウント時にセーブデータが自動で作成されないように設定します。
*
*   @details    この API はセーブデータのマウントを実行する前に設定する必要があります。@n
*               設定は API を実行したプロセス内で有効となります。@n
*               この API はセーブデータフォーマットの移行が完了次第削除されます。
*
*   @return     なし
*
*/
void DisableAutoSaveDataCreation() NN_NOEXCEPT;

//! @}

}} // namespace nn::fs

