﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {

/**
*   @brief      SDMMC 制御ポートを表す列挙型です。
*/
    enum SdmmcPort
    {
        SdmmcPort_Mmc0,                         //!< 本体保存メモリー制御ポート 0 番を指定します。
        SdmmcPort_SdCard0,                      //!< SD カード制御ポート 0 番を指定します。
        SdmmcPort_GcAsic0                       //!< GC ASIC 制御ポート 0 番を指定します。
    };

/**
*   @brief      SDMMC 制御ポートの Speed Mode を表す列挙型です。
*/
    enum SdmmcSpeedMode {
        SdmmcSpeedMode_Unknown,                 //!< 未知
        SdmmcSpeedMode_MmcIdentification,       //!< MMC Identification
        SdmmcSpeedMode_MmcLegacySpeed,          //!< MMC Legacy Speed
        SdmmcSpeedMode_MmcHighSpeed,            //!< MMC High Speed
        SdmmcSpeedMode_MmcHs200,                //!< MMC HS200
        SdmmcSpeedMode_MmcHs400,                //!< MMC HS400
        SdmmcSpeedMode_SdCardIdentification,    //!< SD カード Identification
        SdmmcSpeedMode_SdCardDefaultSpeed,      //!< SD カード Default Speed
        SdmmcSpeedMode_SdCardHighSpeed,         //!< SD カード High Speed
        SdmmcSpeedMode_SdCardSdr12,             //!< SD カード SDR12
        SdmmcSpeedMode_SdCardSdr25,             //!< SD カード SDR25
        SdmmcSpeedMode_SdCardSdr50,             //!< SD カード SDR50
        SdmmcSpeedMode_SdCardSdr104,            //!< SD カード SDR104
        SdmmcSpeedMode_SdCardDdr50,             //!< SD カード DDR50
        SdmmcSpeedMode_GcAsicFpgaSpeed,         //!< GC ASIC FPGA
        SdmmcSpeedMode_GcAsicSpeed              //!< GC ASIC
    };

/**
*   @brief      SDMMC 制御ポートのバス幅を表す列挙型です。
*/
    enum SdmmcBusWidth {
        SdmmcBusWidth_Unknown,                  //!< 未知
        SdmmcBusWidth_1Bit,                     //!< 1 bit
        SdmmcBusWidth_4Bit,                     //!< 4 bit
        SdmmcBusWidth_8Bit                      //!< 8 bit
    };

/**
*   @brief      SDMMC 制御ポートの接続状態を取得します。
*
*   @details    SDMMC 制御ポートの Speed Mode とバス幅を取得します。@n
*               取得される Speed Mode とバス幅は SoC の確立している接続の制御状態を示し、 @n
*               nn::fs::GetMmcSpeedMode(), nn::fs::GetSdCardSpeedMode() で取得される記録メディアの Speed Mode とは必ずしも一致しません。@n
*               規格に準拠していない記録メディアでは、確立している接続状態と異なる Speed Mode を示すものがあります。@n
*
*   @return     接続状態が返ります。
*
*   @pre
*       - SDMMC 制御ポートが何らかのマウント処理により Activate 済みである。
*
*/
Result GetSdmmcConnectionStatus(SdmmcSpeedMode* pOutSdmmcSpeedMode, SdmmcBusWidth* pOutSdmmcBusWidth, SdmmcPort sdmmcPort) NN_NOEXCEPT;

/**
*   @brief      SDMMC 制御を一時停止します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result SuspendSdmmcControl() NN_NOEXCEPT;

/**
*   @brief      SDMMC 制御を再開します。
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*/
Result ResumeSdmmcControl() NN_NOEXCEPT;

}} // namespace nn::fs
