﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace fs {

typedef char PathChar;

namespace StringTraits
{
    // パス区切り文字
    static const PathChar DirectorySeparator = '/';

    // カレント、親ディレクトリ判定用
    static const PathChar Dot = '.';

    // 終端文字
    static const PathChar NullCharactor = '\0';
}

class PathTool
{
public:
    /**
    * @brief        指定した文字が区切り文字かどうかを取得します。
    *
    * @param[in]    ch  文字
    *
    * @return       区切り文字かどうか
    *
    * @details      指定した文字が区切り文字かどうかを取得します。
    */
    static bool IsSeparator(PathChar ch) NN_NOEXCEPT
    {
        return (ch == StringTraits::DirectorySeparator);
    }

    /**
    * @brief        終端文字かどうかを判定します。
    *
    * @param[in]    ch  文字
    *
    * @return       区切り文字かどうか
    */
    static bool IsNul(PathChar ch)
    {
        return (ch == StringTraits::NullCharactor);
    }

    /**
    * @brief        カレントディレクトリを示す文字かどうかを取得します。
    *
    * @param[in]    pDirectoryName  ディレクトリ名
    *
    * @return       カレントディレクトリかどうか
    *
    * @details      カレントディレクトリを示す文字かどうかを取得します。
    */
    static bool IsCurrentDirectory(const PathChar* pDirectoryName) NN_NOEXCEPT
    {
        if( pDirectoryName[0] != StringTraits::Dot )
        {
            return false;
        }
        if ((pDirectoryName[1] != StringTraits::NullCharactor)
         && (pDirectoryName[1] != StringTraits::DirectorySeparator))
        {
            return false;
        }
        return true;
    }

    /**
    * @brief        親ディレクトリを示す文字かどうかを取得します。
    *
    * @param[in]    pDirectoryName  ディレクトリ名
    *
    * @return       カレントディレクトリかどうか
    *
    * @details 親ディレクトリを示す文字かどうかを取得します。
    */
    static bool IsParentDirectory(const PathChar* pDirectoryName) NN_NOEXCEPT
    {
        if( pDirectoryName[0] != StringTraits::Dot )
        {
            return false;
        }
        if( pDirectoryName[1] != StringTraits::Dot )
        {
            return false;
        }
        if( (pDirectoryName[2] != StringTraits::NullCharactor)
         && (pDirectoryName[2] != StringTraits::DirectorySeparator) )
        {
            return false;
        }
        return true;
    }

    /**
    * @brief パスの正規化を行います。
    *
    *        正規化を行うとパス文字列は以下のように変化します。
    *
    *        ・"/" => "/"
    *        ・"/." => "/"
    *        ・"////" => "/"
    *        ・"/dir1" => "/dir1"
    *        ・"/dir1///" => "/dir1"
    *        ・"/dir1/././//" => "/dir1"
    *        ・"/dir1/dir2/.." => "/dir1"
    *
    *        以下の場合にはエラーを返します。
    *
    *        ・先頭が"/"から開始していない。
    *        ・ルートディレクトリよりも親のディレクトリを指定した。
    *
    * @param[out] outValue 正規化後のパス
    * @param[out] outLength 正規化後のパスの文字数(ヌル終端文字を含まない)
    * @param[in] pSrc 正規化前のパス
    * @param[in] outBufLength 正規化後のパスを格納するバッファサイズ
    *
    * @return 関数の処理結果を返します。
    */
    static Result Normalize(char* outValue, size_t* outLength, const char* pSrc, size_t outBufLength) NN_NOEXCEPT
    {
        return Normalize(outValue, outLength, pSrc, outBufLength, false);
    }

    /**
    * @brief パスの正規化を行います。
    *
    *        正規化を行うとパス文字列は以下のように変化します。
    *
    *        ・"/" => "/"
    *        ・"/." => "/"
    *        ・"////" => "/"
    *        ・"/dir1" => "/dir1"
    *        ・"/dir1///" => "/dir1"
    *        ・"/dir1/././//" => "/dir1"
    *        ・"/dir1/dir2/.." => "/dir1"
    *
    *        以下の場合にはエラーを返します。
    *
    *        ・先頭が"/"から開始していない。
    *        ・ルートディレクトリよりも親のディレクトリを指定した。
    *
    * @param[out] outValue 正規化後のパス
    * @param[out] outLength 正規化後のパスの文字数(ヌル終端文字を含まない)
    * @param[in] pSrc 正規化前のパス
    * @param[in] outBufLength 正規化後のパスを格納するバッファサイズ
    * @param[in] isUncPreserved 先頭の連続した区切り文字を UNC とみなして保存するかどうか
    *
    * @return 関数の処理結果を返します。
    */
    static Result Normalize(char* outValue, size_t* outLength, const char* pSrc, size_t outBufLength, bool isUncPreserved) NN_NOEXCEPT;

    /**
    * @brief パスが正規化済みかどうか判定します。
    *
    * @param[out] outNormalized 正規化済みかどうか
    * @param[in] path パス
    *
    * @return 関数の処理結果を返します。
    *
    * @details 連続した区切り文字 ("//")、カレントディレクトリ (".")、親ディレクトリ ("..") を含む場合や
    *          区切り文字で終わる場合を非正規化パスと判定します。
    *          ドライブレター 1 文字とコロンで始まる Windows 形式のパスは
    *          上記の場合であっても正規化パスとみなします。
    */
    static Result IsNormalized(bool* outNormalized, const char* path) NN_NOEXCEPT;

    /**
    * @brief いずれかのパスが他方のサブパスかどうか判定します。
    *
    * @param[in] pLhs パス
    * @param[in] pRhs パス
    *
    * @return 関数の処理結果を返します。
    *
    * @pre パスが正規化済み
    */
    static bool IsSubpath(const char* pLhs, const char* pRhs) NN_NOEXCEPT;
};

}}

