﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_OperateRange.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace fs {

    /**
    * @brief    ファイルシステムドライバ層における、ストレージ領域操作のためのインターフェースです。
    *
    * @details  offset と size が示す領域が重複しない Read() と Write() の呼び出しはスレッドセーフである必要があります。
    */
    class IStorage
    {
    public:
        virtual ~IStorage() NN_NOEXCEPT {}
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT = 0;
        virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
        {
            NN_UNUSED(offset);
            NN_UNUSED(buffer);
            NN_UNUSED(size);
            return fs::ResultUnsupportedOperation();
        }
        virtual Result Flush() NN_NOEXCEPT = 0;
        virtual Result SetSize(int64_t size) NN_NOEXCEPT
        {
            NN_UNUSED(size);
            return fs::ResultUnsupportedOperation();
        }
        virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT = 0;
        virtual Result OperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
        }

        // 入力バッファと出力バッファが不要な場合の OperateRange
        Result OperateRange(OperationId operationId, int64_t offset, int64_t size) NN_NOEXCEPT
        {
            NN_RESULT_DO(OperateRange(nullptr, 0, operationId, offset, size, nullptr, 0));
            NN_RESULT_SUCCESS;
        }

        static inline bool CheckAccessRange(int64_t offset, int64_t size, int64_t storageSize) NN_NOEXCEPT
        {
            auto isValidRange = true
                && offset >= 0
                && size >= 0
                && size <= storageSize
                && offset <= (storageSize - size);
            return isValidRange;
        }

        static inline bool CheckAccessRange(int64_t offset, size_t size, int64_t storageSize) NN_NOEXCEPT
        {
            int64_t castedSize = static_cast<int64_t>(size);
            return CheckAccessRange(offset, castedSize, storageSize);
        }

        static inline bool CheckOffsetAndSize(int64_t offset, int64_t size) NN_NOEXCEPT
        {
            auto isValid = true
                && offset >= 0
                && size >= 0
                && offset <= (offset + size);
            return isValid;
        }

        static inline bool CheckOffsetAndSize(int64_t offset, size_t size) NN_NOEXCEPT
        {
            int64_t castedSize = static_cast<int64_t>(size);
            return CheckOffsetAndSize(offset, castedSize);
        }
    };

}}
