﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace fs {

    /**
     *   @brief      ファイルシステムサーバから通知を受け取る設定を行うためのインターフェイスです。
     */
    class IEventNotifier
    {
    public:
        virtual ~IEventNotifier() NN_NOEXCEPT {}

        /**
         *   @brief      イベントにファイルシステムサーバからの通知を紐づけます。
         *
         *   @details    インターフェイスオブジェクトを破棄すると通知の紐づけは無効化になります。その場合、イベントオブジェクトは破棄されません。@n
         *               使用済みのイベントオブジェクトの破棄は必ず nn::os::DestroySystemEvent() で行ってください。@n
         *               この関数で紐づけられるイベントオブジェクトの実体はインターフェイスオブジェクト毎に 1 つだけです。@n
         *               複数のイベントオブジェクトを個別に操作する場合は、複数のインターフェイスオブジェクトを取得し、そのそれぞれでイベントの紐づけを行ってください。
         *
         *   @param[in]  systemEvent       通知を受け取るイベントオブジェクト
         *   @param[in]  clearMode         イベントのクリアモード
         */
        Result BindEvent(nn::os::SystemEventType* systemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(systemEvent);
            return DoBindEvent(systemEvent, clearMode);
        }

    private:
        virtual Result DoBindEvent(nn::os::SystemEventType* systemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT = 0;
    };

}} // namespace nn::fs
