﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ゲームカードの検査関連で使用するファイルシステム API の宣言
*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Types.h>

namespace nn { namespace fs {

//! @name ゲームカード関連 API
//! @{

/**
 * @brief   GameCardASIC の FW サイズを表す定数です。
 */
const size_t GameCardAsicFwSize = 30 * 1024;

/**
*   @brief      ゲームカードドライバを終了します
*
*   @return     なし
*
*/

void FinalizeGameCardDriver() NN_NOEXCEPT;

/**
*   @brief      RMA 用の FW を流してロット番号などの情報を取得する
*
*   @param[out] pOutValue   取得した GameCardAsic の情報
*   @param[in]  fwBuffer    GameCardAsic の FW の格納バッファ
*   @param[in]  partition   GameCardAsic の FW のバッファサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed
*
*   @pre
*       - fwBufferSize == GameCardAsicFwSize である
*
*/
Result GetGameCardAsicInfo(nn::gc::RmaInformation* pOutValue, const void* fwBuffer, size_t fwBufferSize) NN_NOEXCEPT;

/**
*   @brief      挿入されているゲームカードの ID 情報を取得する
*
*   @param[out]  pOutValue   取得したゲームカードの ID セット
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed
*
*   @pre
*       - ゲームカードが挿入されている
*
*/

Result GetGameCardIdSet(nn::gc::GameCardIdSet* pOutValue) NN_NOEXCEPT;

/**
*   @brief      挿入されている製品版 Writable カードの Device 証明書を取得する
*
*   @param[out]  buffer     Device 証明書 の格納バッファ
*   @param[out]  bufferSize Device 証明書 の格納バッファサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed
*
*   @pre
*       - ゲームカードが挿入されている
*
*/
Result GetGameCardDeviceIdForProdCard(void* buffer, size_t bufferSize, void* cardHeaderForDev, size_t cardHeaderForDevSize) NN_NOEXCEPT;
Result EraseAndWriteParamDirectly(void* buffer, size_t bufferSize) NN_NOEXCEPT;
Result ReadParamDirectly(void* buffer, size_t bufferSize) NN_NOEXCEPT;
Result ForceEraseGameCard() NN_NOEXCEPT;


//! @}

}} // namespace nn::fs
