﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  ゲームカード関連のファイルシステム API の宣言
*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {
    class IStorage;
}}

namespace nn { namespace fs {
    class IEventNotifier;
}}


namespace nn { namespace fs {

//! @name ゲームカード関連 API
//! @{

/**
*   @brief ゲームカードのマウントのためのパーティション識別子です。
*/
enum class GameCardPartition
{
    Update,
    Normal,
    Secure,
    Logo, // FS 内部実装用
};

/**
*   @brief ゲームカードの raw アクセスのためのパーティション識別子です。
*/
enum class GameCardPartitionRaw
{
    NormalReadOnly,
    SecureReadOnly,
    RootWriteOnly,  //! Key area を含む全領域を Write only で開きます
};

/**
*   @brief ゲームカードアプリの属性です。
*/
enum class GameCardAttribute : uint8_t
{
    AutoBootFlag      = 1,
    HistoryEraseFlag  = 1 << 1,
    RepairToolFlag    = 1 << 2
};

//! @brief ゲームカードのハンドルです。
typedef uint32_t GameCardHandle;

//! @brief ゲームカードのアップデートパーティションのバージョン値です。
typedef uint32_t UpdateVersion;

struct GameCardUpdatePartitionInfo
{
    UpdateVersion version;
    Bit64 id;
};

//! @brief ゲームカードのエラー情報です。この構造体は廃止されます。GameCardErrorReportInfo を利用してください。
struct GameCardErrorInfo
{
    uint16_t gameCardCrcErrorNum;
    uint16_t reserved1;
    uint16_t asicCrcErrorNum;
    uint16_t reserved2;
    uint16_t refreshNum;
    uint16_t reserved3;
    uint16_t retryLimitOutNum;
    uint16_t timeoutRetryNum;
};

//! @brief ゲームカードのエラー情報です。
struct GameCardErrorReportInfo
{
    uint16_t gameCardCrcErrorNum;
    uint16_t reserved1;
    uint16_t asicCrcErrorNum;
    uint16_t reserved2;
    uint16_t refreshNum;
    uint16_t reserved3;
    uint16_t retryLimitOutNum;
    uint16_t timeoutRetryNum;
    uint16_t asicReinitializeFailureDetail;
    uint16_t insertionCount;
    uint16_t removalCount;
    uint16_t asicReinitializeNum;
    uint32_t initializeCount;
    uint16_t asicReinitializeFailureNum;
    uint16_t awakenFailureNum;
    uint16_t reserved5;
    uint16_t refreshSucceededCount;
    uint32_t lastReadErrorPageAddress;
    uint32_t lastReadErrorPageCount;
    uint32_t awakenCount;
    uint32_t readCountFromInsert;
    uint32_t readCountFromAwaken;
    uint8_t  reserved6[8];
};

/**
* @brief   ゲームカード のデバイス証明書のサイズを表す定数です。
*/
const size_t GameCardDeviceCertificateSize = 512;

/**
* @brief   ゲームカード のカードイメージハッシュのサイズを表す定数です。
*/
const size_t GameCardImageHashSize = 32;

/**
 * @brief   ゲームカードの CID のサイズを表す定数です。
 */
const size_t GameCardCidSize = 16;

/**
* @brief   ゲームカードの DeviceID のサイズを表す定数です。
*/
const size_t GameCardDeviceIdSize = 16;


/**
*   @brief      ゲームカードの挿抜通知設定のためのインターフェイスを取得します。
*
*   @param[out] outValue   取得されたインターフェイス
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*   @pre
*       - outValue が有効なメモリを指している。
*/
Result OpenGameCardDetectionEventNotifier(std::unique_ptr<IEventNotifier>* outValue) NN_NOEXCEPT;

/**
*   @brief      ゲームカードの挿抜状態を取得します。
*
*   @return     挿入されている場合は true、抜かれている場合は false が返ります。
*/
bool IsGameCardInserted() NN_NOEXCEPT;


/**
*   @brief      ゲームカードのアップデートパーティションの情報を取得します。
*
*   @param[out] pOutUpdatePartitionInfo   取得したアップデートパーティションの情報の格納先
*   @param[in]  handle          GetGameCardHandle で取得したゲームカードアクセス用のハンドル
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*       - handle が GetGameCardHandle() で取得したものである。
*/
Result GetGameCardUpdatePartitionInfo(GameCardUpdatePartitionInfo* pOutUpdatePartitionInfo, GameCardHandle handle) NN_NOEXCEPT;

/**
*   @brief      ゲームカードのハンドルを取得します。
*
*   @details    現在挿入されているゲームカードのハンドルを取得します。
*               取得したハンドルはゲームカードが抜去されるまで有効です。
*
*   @param[out] outValue   取得したハンドルの格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*/
Result GetGameCardHandle(GameCardHandle* pOutValue) NN_NOEXCEPT;


/**
*   @brief      ゲームカードのパーティションをマウントします。
*
*   @param[in]  name       マウント名
*   @param[in]  handle     ゲームカードのハンドル
*   @param[in]  partition  パーティション識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - handle が GetGameCardHandle() で取得したものである。
*
*/
Result MountGameCardPartition(const char* name, GameCardHandle handle, GameCardPartition partition) NN_NOEXCEPT;



/**
*   @brief      ゲームカードの raw アクセスのためのアクセサを取得します。
*
*   @param[out] outValue   取得したインターフェースの格納先
*   @param[in]  handle     ゲームカードのハンドル
*   @param[in]  partition  パーティション識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultPartitionNotFound         利用可能なパーティションが見つかりません。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - outValue が有効なメモリを指している。
*       - handle が GetGameCardHandle() で取得したものである。
*/
Result OpenGameCardPartition(std::unique_ptr<nn::fs::IStorage>* outValue, GameCardHandle handle, GameCardPartitionRaw partition) NN_NOEXCEPT;


/**
*   @brief      現在挿入されているカードの GameCardAttribute を取得します。
*
*   @param[out] pOutValue  GameCardAttribute の格納先
*   @param[in]  handle     ゲームカードのハンドル
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - pOutValue が有効なメモリを指している。
*       - handle が GetGameCardHandle() で取得したものである。
*/

Result GetGameCardAttribute(GameCardAttribute* pOutValue, GameCardHandle handle) NN_NOEXCEPT;

/**
*   @brief      現在挿入されている ゲームカードのデバイス証明書を取得します。
*
*   @param[out] buffer     ゲームカードのデバイス証明書の格納先バッファ
*   @param[in]  bufferSize 確保してある buffer のサイズ
*   @param[in]  handle     ゲームカードのハンドル
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - buffer が有効なメモリを指している。
*       - handle が GetGameCardHandle() で取得したものである。
*       - GameCardPartition Secure で Open あるいは Mount された後である。
*       - bufferSize >= GameCardDeviceCertificateSize である。
*/

Result GetGameCardDeviceCertificate(void* buffer, size_t bufferSize, GameCardHandle handle) NN_NOEXCEPT;

/**
*   @brief      現在挿入されている ゲームカードのカードイメージハッシュを取得します。
*
*   @param[out] buffer     ゲームカードのカードイメージハッシュの格納先バッファ
*   @param[in]  bufferSize 確保してある buffer のサイズ
*   @param[in]  handle     ゲームカードのハンドル
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - buffer が有効なメモリを指している。
*       - handle が GetGameCardHandle() で取得したものである。
*       - bufferSize == GameCardImageHashSize である。
*/

Result GetGameCardImageHash(void* buffer, size_t bufferSize, GameCardHandle handle) NN_NOEXCEPT;

/**
*   @brief ゲームカードのサイズ識別子です。
*/
enum class GameCardSize
{
    Size1GB  = 1,
    Size2GB  = 2,
    Size4GB  = 4,
    Size8GB  = 8,
    Size16GB = 16,
    Size32GB = 32,
};

/**
*   @brief ゲームカードの動作周波数識別子です。
*/
enum class GameCardClockRate
{
    ClockRate25MHz = 25,
    ClockRate50MHz = 50,
};

/**
*   @brief      ゲームカードを消去し、指定したサイズで初期化します。
*
*   @param[in]  size  ゲームカードのサイズ識別子
*   @param[in]  normalAreaSize ノーマルエリアのバイトサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultPartitionNotFound         利用可能なパーティションが見つかりません。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*/
Result EraseGameCard(GameCardSize size, uint64_t normalAreaSize) NN_NOEXCEPT;

/**
*   @brief      開発用ゲームカードへの書き込みを行います。
*
*   @param[in]  offset     ゲームカードのバイトオフセット
*   @param[in]  buffer     書き込みを行うバッファ
*   @param[in]  size       書き込みを行うサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - offset >= 0
*       - buffer が有効なメモリを指している。
*       - RootWriteOnly で  OpenGameCardPartition した後である。
*       - buffer が nn::dd::DeviceAddressSpaceMemoryRegionAlignment でアライメントされている。
*       - size が nn::gc::GcPageSize の倍数、かつ nn::dd::DeviceAddressSpaceMemoryRegionAlignment の倍数である。
*       - Write 時に Verify したい場合は SetVerifyWriteEnalbleFlag() で先にフラグを設定する。
*
*   @details    与えられたバッファを GC ドライバで直接使用するため、 API 実行後 buffer の中身は書き換えられています。
*/
Result WriteToGameCard(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;


/**
*   @brief      開発用ゲームカードへの書き込み時に  Verify を行うフラグを設定します。
*
*   @param[in]  isVerifyWriteEnable             開発用ゲームカードへの書き込み時に  Verify を行うフラグ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*/
Result SetVerifyWriteEnalbleFlag(bool verifyWriteEnable) NN_NOEXCEPT;

/**
*   @brief      エラー発生時のゲームカード関連の情報を取得します。この API は廃止されます。GetGameCardErrorReportInfo を利用してください。
*
*   @param[out] pOutGameCardErrorInfo   取得したゲームカードエラー情報の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*/
Result GetGameCardErrorInfo(GameCardErrorInfo* pOutGameCardErrorInfo) NN_NOEXCEPT;

/**
*   @brief      エラー発生時のゲームカード関連の詳細情報を取得します。
*
*   @param[out] pOutGameCardErrorReportInfo   取得したゲームカードエラー情報の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*
*/
Result GetGameCardErrorReportInfo(GameCardErrorReportInfo* pOutGameCardErrorReportInfo) NN_NOEXCEPT;

/**
*   @brief      GameCard の CID (ID1/2/3 を連結したもの) を取得します。
*
*   @param[out] pOutBuffer  取得した CID の格納先
*   @param[in]  size        取得した CID を格納する領域のサイズ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @pre        size >= GameCardCidSize
*
*/
Result GetGameCardCid(void* outBuffer, size_t size) NN_NOEXCEPT;

/**
*   @brief      ゲームカードのパーティションがアクセス可能か確認します。
*               パーティション識別子に Secure を指定した場合、副作用としてゲームカードの状態が Secure モードに遷移します。
*
*   @param[in]  handle     ゲームカードのハンドル
*   @param[in]  partition  パーティション識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultGameCardAccessFailed      ゲームカードへのアクセスに失敗しました。
*
*   @pre
*       - handle が GetGameCardHandle() で取得したものである。
*
*/
Result CheckGameCardPartitionAvailability(GameCardHandle handle, GameCardPartition partition) NN_NOEXCEPT;

/**
*   @brief      GameCardIdInfo の構造体を取得します。
*               API が呼び出された時点で挿入されているゲームカードの情報を取得できます。
*
*   @param[out] pOutGameCardIdInfo   取得したゲームカード ID 情報の格納先
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess                   成功しました。
*   @retval     ResultNotInserted               ゲームカードが挿入されていません
*
*/
Result GetGameCardDeviceId(void* outBuffer, size_t size) NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
