﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/os/os_Mutex.h>
#include <nn/util/util_Optional.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>


namespace nn { namespace fs { namespace fsa {
class IFile;
class IFileSystem;
}}}

namespace nn { namespace fs {

/**
* @brief IFile を IStorage にするアダプタです。
*/
class FileStorage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(FileStorage);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @param[in]    pBaseFile     基となるIFile
    *
    * @pre          pBaseFile は有効な IFile を指している
    *
    * @details      pBaseFile はインスタンスが破棄されるまで参照されます。
    */
    explicit FileStorage(nn::fs::fsa::IFile* pBaseFile) NN_NOEXCEPT;

    explicit FileStorage(std::shared_ptr<nn::fs::fsa::IFile> pSharedBaseFile) NN_NOEXCEPT;

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

protected:
    FileStorage() NN_NOEXCEPT;

protected:
    void SetFile(fs::fsa::IFile* pFile) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFile);
        NN_SDK_REQUIRES(m_pFile == nullptr);
        m_pFile = pFile;
    }

private:
    static const int64_t SizeNotInitialized = -1;

private:
    Result UpdateSize() NN_NOEXCEPT;

private:
    fs::fsa::IFile* m_pFile;
    std::shared_ptr<fsa::IFile> m_pSharedFile;
    int64_t m_Size;
};

/**
* @brief    IFileSystem と path から取得した IFile を IStorage にするアダプタです。
*/
class FileStorageBasedFileSystem : public FileStorage
{
    NN_DISALLOW_COPY(FileStorageBasedFileSystem);

public:
    FileStorageBasedFileSystem() NN_NOEXCEPT;
    Result Initialize(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT;

private:
    std::shared_ptr<nn::fs::fsa::IFileSystem> m_BaseFileSystem;
    std::unique_ptr<nn::fs::fsa::IFile> m_BaseFile;
};

/**
* @brief ファイルハンドルを IStorage にするアダプタです。
*/
class FileHandleStorage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(FileHandleStorage);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @param[in]    handle      基となるファイルハンドル
    *
    *
    * @pre          handle は有効なファイルハンドルである
    *
    * @details      handle はインスタンスが破棄されるまで使用されます。
    */
    explicit FileHandleStorage(nn::fs::FileHandle handle) NN_NOEXCEPT;
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    const nn::fs::FileHandle m_Handle;
    int64_t m_Size;
    os::Mutex m_AccessMutex;
};

}}
