﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  コンテンツストレージ関連 API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace fs {

/**
* @brief コンテンツストレージの識別子です。
*/
enum class ContentStorageId : uint32_t
{
    System = 0,
    User,
    SdCard,
};

/**
*   @brief コンテンツストレージのディレクトリ名です。
*/
const char* const ContentStorageDirectoryName = "Contents";


//! @name コンテンツストレージ関連 API
//! @{
//--------------------------------------------------------------------------

/**
*   @brief      MountContentStorage() でマウントしたファイルシステムを利用するためのマウント名を取得します。
*
*   @param[in]  id         コンテンツストレージの識別子
*/
const char* GetContentStorageMountName(ContentStorageId id) NN_NOEXCEPT;

/**
*   @brief      ID に紐づいたコンテンツストレージをマウントします。
*
*   @param[in]  id         コンテンツストレージの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*
*   @post
*       - GetContentStorageMountName(id) で取得できるマウント名で対象がマウントされている。
*/
Result MountContentStorage(ContentStorageId id) NN_NOEXCEPT;


/**
*   @brief      ID に紐づいたコンテンツストレージをマウントします。
*
*   @param[in]  name       マウント名
*   @param[in]  id         コンテンツストレージの識別子
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。

*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*
*   @post
*       - マウント名 name で対象がマウントされている。
*/
Result MountContentStorage(const char* name, ContentStorageId id) NN_NOEXCEPT;


//! @}

}} // namespace nn::fs
