﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  コンテンツ関連のファイルシステム API の宣言
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_DataId.h>
#include <nn/ncm/ncm_ProgramId.h>

namespace nn { namespace fs {

//! @name コンテンツ関連 API
//! @{

enum ContentType
{
    ContentType_Meta,
    ContentType_Control,
    ContentType_Manual,
    ContentType_Logo,
    ContentType_Data
};

//--------------------------------------------------------------------------
/**
*   @brief      コンテンツ内部を参照するためのファイルシステムをマウントします。
*
*   @param[in]  name            マウント名
*   @param[in]  path            パス
*   @param[in]  id              コンテンツのプログラム ID
*   @param[in]  contentType     コンテンツのタイプ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          path で表されるコンテンツパスが有効ではありません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - path が @link entry_path 有効なパス @endlink である。
*/
Result MountContent(const char* name, const char* path, ncm::ProgramId id, ContentType contentType) NN_NOEXCEPT;
Result MountContent(const char* name, const char* path, ncm::DataId id, ContentType contentType) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      コンテンツ内部を参照するためのファイルシステムをマウントします。
*
*   @details    このオーバーロードではコンテンツメタしか開くことはできず、コンテンツの所有者 ID チェックを行いません。
*
*   @param[in]  name            マウント名
*   @param[in]  path            パス
*   @param[in]  contentType     コンテンツのタイプ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          path で表されるコンテンツパスが有効ではありません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - path が @link entry_path 有効なパス @endlink である。
*       - contentType == ContentType::ContentType_Meta
*/
Result MountContent(const char* name, const char* path, ContentType contentType) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
*   @brief      コンテンツ内部を参照するためのファイルシステムをマウントします。
*
*   @details    このオーバーロードでは対象コンテンツに適用できるパッチコンテンツが存在した場合、
*               パッチを適用した状態でファイルシステムをマウントします。
*
*   @param[in]  name            マウント名
*   @param[in]  id              コンテンツのプログラム ID
*   @param[in]  contentType     コンテンツのタイプ
*
*   @return     処理の結果が返ります。
*   @retval     ResultSuccess               成功しました。
*   @retval     ResultPathNotFound          id に紐づくコンテンツが存在しません。
*
*   @pre
*       - name が @link mount_name 有効なマウント名 @endlink である。
*       - id が有効なプログラム ID である。
*/
Result MountContent(const char* name, ncm::ProgramId id, ContentType contentType) NN_NOEXCEPT;

//! @}

}} // namespace nn::fs
