﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/detail/fs_Log.h>

#if defined(NN_DETAIL_ENABLE_ABORT_MESSAGE)
    #define NN_DETAIL_FS_ENABLE_WARN_ON_DATA_CORRUPTED 1 // NOLINT(preprocessor/const)
#endif // defined(NN_DETAIL_ENABLE_ABORT_MESSAGE)

#if defined(NN_DETAIL_FS_ENABLE_WARN_ON_DATA_CORRUPTED)
    #define NN_DETAIL_FS_WARN_ON_DATA_CORRUPTED NN_DETAIL_FS_WARN
#else // defined(NN_DETAIL_FS_ENABLE_WARN_ON_DATA_CORRUPTED)
    #define NN_DETAIL_FS_WARN_ON_DATA_CORRUPTED(...)
#endif // defined(NN_DETAIL_FS_ENABLE_WARN_ON_DATA_CORRUPTED)

#define NN_FS_DETAIL_RESULT_ABORT(r, force) \
{ \
    if( ::nn::fs::detail::IsAbortNeeded(r) || force ) \
    { \
        ::nn::fs::detail::LogErrorMessage(r, NN_CURRENT_FUNCTION_NAME); \
        NN_ABORT_UNLESS_RESULT_SUCCESS(r); \
    } \
}

// アボートが必要なリザルトのみアボートする
#define NN_FS_RESULT_DO(r) \
{ \
    ::nn::Result nnFsResult = (r); \
    NN_FS_DETAIL_RESULT_ABORT(nnFsResult, false); \
    NN_RESULT_DO(nnFsResult); \
}

#define NN_FS_RESULT_DO_WITH_HANDLER(r, handler) \
{ \
    ::nn::Result nnFsResult = handler(r); \
    NN_FS_DETAIL_RESULT_ABORT(nnFsResult, false); \
    NN_RESULT_DO(nnFsResult); \
}

#define NN_FS_RESULT_THROW(r) \
{ \
    ::nn::Result nnFsResult = (r); \
    NN_FS_DETAIL_RESULT_ABORT(nnFsResult, false); \
    return nnFsResult; \
}

#define NN_FS_RESULT_THROW_UNLESS(condition, r) \
{ \
    if( condition ) {} \
    else \
    { \
        ::nn::Result nnFsResult = (r); \
        NN_FS_DETAIL_RESULT_ABORT(nnFsResult, false); \
        return nnFsResult; \
    } \
}

// ResultSuccess 以外はアボートする
#define NN_FS_ABORT_UNLESS_RESULT_SUCCESS(r) \
{ \
    ::nn::Result nnFsResult = (r); \
    NN_FS_DETAIL_RESULT_ABORT(nnFsResult, true); \
}

// condition が偽のときに、指定の Result で NN_FS_ABORT_UNLESS_RESULT_SUCCESS する
#define NN_FS_ABORT_UNLESS_WITH_RESULT(condition, r, ...) \
    do \
    { \
        if(!NN_STATIC_CONDITION(condition)) \
        { \
            NN_DETAIL_FS_ERROR("Abort: " NN_MACRO_STRINGIZE(condition) "Message: \n  " __VA_ARGS__); \
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(r); \
        } \
    } while (NN_STATIC_CONDITION(0))

// GC はハードウェア要因によりデータ化け発生時はカード抜けを疑う必要があり、そのマクロ（副作用に注意）
#define NN_DETAIL_FS_DATA_CORRUPTED_RETRY_RESULT_DO(_statement) \
{ \
    static const int _MaxGameCardReadRetryNum = 2; \
    Result _retry_result = ResultSuccess(); \
    for (int _retry_i = 0; _retry_i < _MaxGameCardReadRetryNum; _retry_i++) \
    { \
        _retry_result = _statement; \
        if (fs::ResultDataCorrupted::Includes(_retry_result)) \
        { \
            NN_DETAIL_FS_WARN_ON_DATA_CORRUPTED("DataCorrupted Failure (retry count: %d), 0x%08x at %s() - %d\n", _retry_i, _retry_result.GetInnerValueForDebug(), __FUNCTION__, __LINE__); \
            continue; \
        } \
        break; \
    } \
    NN_RESULT_DO(_retry_result); \
}

#if NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT

    #define NN_FS_DETAIL_RESULT_THROW_IMPL(r) \
        do \
        { \
            NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
            NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(_nn_result_throw_impl_source_info); \
            NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
            ::nn::Result nnFsResult = ::nn::result::detail::MakeResultAutoConverterWithSourceInfo(r, _nn_result_throw_impl_source_info); \
            NN_FS_DETAIL_RESULT_ABORT(nnFsResult, false); \
            return nnFsResult; \
        } while (NN_STATIC_CONDITION(0))

#else

    #define NN_FS_DETAIL_RESULT_THROW_IMPL(r) \
        do \
        { \
            ::nn::Result nnFsResult = ::nn::result::detail::MakeResultAutoConverter(r); \
            NN_FS_DETAIL_RESULT_ABORT(nnFsResult, false); \
            return nnFsResult; \
        } while (NN_STATIC_CONDITION(0))

#endif

#define NN_FS_RESULT_TRY(r) NN_RESULT_TRY(r)

#define NN_FS_RESULT_CATCH(type) NN_RESULT_CATCH(type)

#define NN_FS_RESULT_CATCH_ALL NN_RESULT_CATCH_ALL

#define NN_FS_RESULT_RETHROW \
                do \
                { \
                    NN_DETAIL_RESULT_MARK_AS_RETHROW; \
                    NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW(_nn_result_try_temporary); \
                    NN_FS_DETAIL_RESULT_THROW_IMPL(_nn_result_try_temporary); \
                } while (NN_STATIC_CONDITION(0))

#define NN_FS_RESULT_END_TRY \
            } \
            else \
            { \
                NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
                NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(_nn_result_end_try_impl_source_info); \
                NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
                NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS(_nn_result_try_temporary, _nn_result_end_try_impl_source_info); \
                NN_FS_DETAIL_RESULT_THROW_IMPL(_nn_result_try_temporary); \
            } \
        } \
    }

namespace nn { namespace fs { namespace detail {

bool IsAbortNeeded(nn::Result result) NN_NOEXCEPT;
void LogErrorMessage(nn::Result result, const char* functionName) NN_NOEXCEPT;

}}}
