﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/util/util_IntUtil.h>


namespace nn { namespace fs { namespace detail {

struct BufferRegion
{
public:
    static const BufferRegion Null;

public:
    int64_t offset;
    size_t size;

    BufferRegion() NN_NOEXCEPT
        : offset(0)
        , size(0)
    {
    }
    BufferRegion(int64_t offset, size_t size) NN_NOEXCEPT
        : offset(offset)
        , size(size)
    {
        // size は書き換えられるので意味ない気がする
        if (NN_STATIC_CONDITION(sizeof(size_t) == 8))
        {
            NN_ABORT_UNLESS(size <= INT64_MAX);
        }
    }

public:
    static BufferRegion GetInclusion(const BufferRegion& region1, const BufferRegion& region2) NN_NOEXCEPT
    {
        int64_t offset = std::min(region1.offset, region2.offset);
        int64_t endOffset = std::max(region1.GetEndOffset(), region2.GetEndOffset());
        int64_t newSize = endOffset - offset;
        NN_ABORT_UNLESS(util::IsIntValueRepresentable<size_t>(newSize));
        return BufferRegion(offset, static_cast<size_t>(newSize));
    }

    static BufferRegion GetUnion(const BufferRegion& region1, const BufferRegion& region2) NN_NOEXCEPT
    {
        if (!region1.Intersects(region2))
        {
            return Null;
        }
        return GetInclusion(region1, region2);
    }

    static BufferRegion GetIntersection(const BufferRegion& region1, const BufferRegion& region2) NN_NOEXCEPT
    {
        if (!region1.Intersects(region2))
        {
            return Null;
        }
        int64_t offset = std::max(region1.offset, region2.offset);
        int64_t endOffset = std::min(region1.GetEndOffset(), region2.GetEndOffset());
        int64_t newSize = endOffset - offset;
        NN_ABORT_UNLESS(util::IsIntValueRepresentable<size_t>(newSize));
        return BufferRegion(offset, static_cast<size_t>(newSize));
    }

    static bool HasIntersection(const BufferRegion& region1, const BufferRegion& region2) NN_NOEXCEPT
    {
        if (region1.GetEndOffset() < region2.offset)
        {
            return false;
        }
        if (region2.GetEndOffset() < region1.offset)
        {
            return false;
        }
        return true;
    }

public:
    int64_t GetEndOffset() const NN_NOEXCEPT
    {
        return static_cast<int64_t>(offset + size);
    }

    BufferRegion ExpandAndAlign(size_t alignment) const NN_NOEXCEPT
    {
        int64_t newOffset = util::align_down(offset, alignment);
        int64_t newEndOffset = util::align_up(GetEndOffset(), alignment);
        int64_t newSize = newEndOffset - newOffset;
        NN_ABORT_UNLESS(util::IsIntValueRepresentable<size_t>(newSize));
        return BufferRegion(newOffset, static_cast<size_t>(newSize));
    }

    BufferRegion ShrinkAndAlign(size_t alignment) const NN_NOEXCEPT
    {
        int64_t newOffset = util::align_up(offset, alignment);
        int64_t newEndOffset = util::align_down(GetEndOffset(), alignment);
        int64_t newSize = newEndOffset - newOffset;
        NN_ABORT_UNLESS(util::IsIntValueRepresentable<size_t>(newSize));
        return BufferRegion(newOffset, static_cast<size_t>(newSize));
    }

    BufferRegion GetEndRegionWithSizeLimit(size_t sizeLimit) const NN_NOEXCEPT
    {
        if (sizeLimit >= size)
        {
            return *this;
        }
        int64_t newOffset = GetEndOffset() - sizeLimit;
        return BufferRegion(newOffset, sizeLimit);
    }

    BufferRegion GetInclusion(const BufferRegion& other) const NN_NOEXCEPT
    {
        return BufferRegion::GetInclusion(*this, other);
    }

    BufferRegion GetUnion(const BufferRegion& other) const NN_NOEXCEPT
    {
        return BufferRegion::GetUnion(*this, other);
    }

    BufferRegion GetIntersection(const BufferRegion& other) const NN_NOEXCEPT
    {
        return BufferRegion::GetIntersection(*this, other);
    }

    bool Intersects(const BufferRegion& other) const NN_NOEXCEPT
    {
        return BufferRegion::HasIntersection(*this, other);
    }

    bool Includes(const BufferRegion& other) const NN_NOEXCEPT
    {
        return offset <= other.offset && other.GetEndOffset() <= GetEndOffset();
    }

    bool IncludesOffset(int64_t theOffset) const NN_NOEXCEPT
    {
        return offset <= theOffset && theOffset <= GetEndOffset();
    }

    bool operator==(const BufferRegion& rhs) const NN_NOEXCEPT
    {
        return offset == rhs.offset && size == rhs.size;
    }

    bool operator!=(const BufferRegion& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }
};

}}}  // namespace nn::fs::detail
