﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Device Tree のノードの検索・探索・情報取得用 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dt/dt_Types.h>

namespace nn { namespace dt {

//! @name ノードの検索・探索・情報取得用 API
//! @{

/**
 * @brief       指定されたパスにあるノードを検索します。
 *
 * @param[out]  pOutNode  検索結果のノード
 * @param[in]   path      検索キーとなるパス
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     検索パスを path に指定して、検索結果のノードを pOutNode に格納します。
 *              path として無効な文字列を与えた場合は ResultInvalidArgument が返ります。
 *              path にノードが存在しなかった場合は、ResultNodeNotExist が返ります。
 *
 */
nn::Result FindNodeByPath(Node* pOutNode, const char* path) NN_NOEXCEPT;

/**
 * @brief       指定された phandle に対応するノードを検索します。
 *
 * @param[out]  pOutNode  検索結果のノード
 * @param[in]   phandle   検索キーとなる phandle
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 * @endretresult
 *
 * @details     phandle に検索キーを指定して、検索結果のノードを pOutNode に格納します。
 *              無効な phandle を与えた場合には ResultInvalidArgument が返ります。
 *
 */
nn::Result FindNodeByPHandle(Node* pOutNode, PHandle phandle) NN_NOEXCEPT;

/**
 * @brief       compatible なノードの数を取得します。
 *
 * @param[out]  pOutCount   ノードの数
 * @param[in]   compatible  compatible プロパティの文字列
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     引数 compatible に文字列を指定して、
 *              それに完全一致する文字列を compatible プロパティに含むノードの数を pOutCount に格納します。
 *
 */
nn::Result GetCompatibleNodeCount(int* pOutCount, const char* compatible) NN_NOEXCEPT;

/**
 * @brief       compatible なノードのリストを取得します。
 *
 * @param[out]  pOutNodeList  ノードのリスト
 * @param[out]  pOutCount     pOutNodeList に格納したノードの数
 * @param[in]   maxCount      pOutNodeList に格納できるノードの最大数
 * @param[in]   compatible    compatible プロパティの文字列
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultBufferTooSmall}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     引数 compatible に文字列を指定して、
 *              それに完全一致する文字列を compatible プロパティに含むノードのリストを取得し、
 *              pOutNodeList に格納します。
 *
 */
nn::Result ListCompatibleNode(Node* pOutNodeList, int* pOutCount, int maxCount, const char* compatible) NN_NOEXCEPT;

/**
 * @brief       親ノードを取得します。
 *
 * @param[out]  pOutNode      親ノード
 * @param[in]   pCurrentNode  基準となるノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     pCurrentNode の親ノードを pOutNode に格納します。
 *              pCurrentNode にルートノードを指定した場合は ResultNodeNotExist が返ります。
 *
 */
nn::Result GetParentNode(Node* pOutNode, const Node* pCurrentNode) NN_NOEXCEPT;

/**
 * @brief       指定した名前の子ノードを取得します。
 *
 * @param[out]  pOutNode      子ノード
 * @param[in]   pCurrentNode  基準となるノード
 * @param[in]   name          子ノードの名前
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     name に指定した名前を持つ、pCurrentNode の子ノードを pOutNode に格納します。
 *
 */
nn::Result GetChildNodeByName(Node* pOutNode, const Node* pCurrentNode, const char* name) NN_NOEXCEPT;

/**
 * @brief       子ノードの数を取得します。
 *
 * @param[out]  pOutCount     子ノードの数
 * @param[in]   pCurrentNode  基準となるノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     pCurrentNode の子ノードの数を pOutCount に格納します。
 *
 */
nn::Result GetChildNodeCount(int* pOutCount, const Node* pCurrentNode) NN_NOEXCEPT;

/**
 * @brief       子ノードのリストを取得します。
 *
 * @param[out]  pOutNodeList  子ノードのリスト
 * @param[out]  pOutCount     pOutNodeList に格納した子ノードの数
 * @param[in]   maxCount      pOutNodeList に格納できるノードの最大数
 * @param[in]   pCurrentNode  基準となるノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultBufferTooSmall}
 * @endretresult
 *
 * @details     pCurrentNode の子ノードの数を pOutCount に格納します。
 *
 */
nn::Result ListChildNode(Node* pOutNodeList, int* pOutCount, int maxCount, const Node* pCurrentNode) NN_NOEXCEPT;

/**
 * @brief       最初の子ノードを取得します。
 *
 * @param[out]  pOutNode      最初の子ノード
 * @param[in]   pCurrentNode  基準となるノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     pCurrentNode の最初の子ノードを pOutNode に格納します。
 *              子ノードが存在しない場合は ResultNodeNotExist を返します。
 *
 */
nn::Result GetFirstChildNode(Node* pOutNode, const Node* pCurrentNode) NN_NOEXCEPT;

/**
 * @brief       次のノードを取得します。
 *
 * @param[out]  pOutNode      次のノード
 * @param[in]   pCurrentNode  基準となるノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 *   @handleresult{nn::dt::ResultNodeNotExist}
 * @endretresult
 *
 * @details     pCurrentNode と同じ階層にある次のノードを pOutNode に格納します。
 *              該当するノードが無い場合は ResultNodeNotExist を返します。
 *
 */
nn::Result GetNextSiblingNode(Node* pOutNode, const Node* pCurrentNode) NN_NOEXCEPT;

/**
 * @brief       ノードの address-cells を取得します。
 *
 * @param[out]  pOutValue address-cells の値
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 * @endretresult
 *
 * @details     pNode の address-cells を pOutValue に格納します。
 *
 */
nn::Result GetValueOfAddressCells(uint32_t* pOutValue, const Node* pNode) NN_NOEXCEPT;

/**
 * @brief       ノードの size-cells を取得します。
 *
 * @param[out]  pOutValue size-cells の値
 * @param[in]   pNode     ノード
 *
 * @retresult
 *   @handleresult{nn::dt::ResultInvalidArgument}
 *   @handleresult{nn::dt::ResultFdtCorrupted}
 * @endretresult
 *
 * @details     pNode の size-cells を pOutValue に格納します。
 *
 */
nn::Result GetValueOfSizeCells(uint32_t* pOutValue, const Node* pNode) NN_NOEXCEPT;

//! @}

}}
