﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief コールバック登録式のイベントハンドラに関する宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

namespace nn { namespace ddsf {

//! @name イベントハンドラ関連 API
//! @{

/**
* @brief   イベントハンドラのインタフェースクラスです。
*
* @detail
*   イベントハンドラが備えるメソッドを定義するインタフェースクラスです。 @n
*   @ref EventHandlerLooper::RegisterHandler() に本インタフェースを継承するオブジェクトを渡すことで、
*   関連付けた同期オブジェクトがシグナルされた際に所定のハンドラを呼び出させることができます。
*/
class IEventHandler
{
    NN_DISALLOW_COPY(IEventHandler);
    NN_DISALLOW_MOVE(IEventHandler);

    friend class EventHandlerLooper;

public:
    //! @name 生成と破棄
    //! @{

    /**
    * @brief デフォルトコンストラクタです。
    */
    IEventHandler() NN_NOEXCEPT;

protected:
    /**
    * @brief    デストラクタです。
    *
    * @detail
    *   デストラクタです。 @n
    *   本クラスは継承を前提としており、直接の実体化や基底クラスに対する delete を禁止するため protected non-virtual としています。
    */
    ~IEventHandler() NN_NOEXCEPT;

    //! @}

public:

    //! @name イベントハンドラの設定
    //! @{

    /**
    * @brief イベントハンドラオブジェクトを初期化します。
    *
    * @tparam    T  同期オブジェクトの型
    *
    * @param[in]    pSyncObject 同期オブジェクト。 nn::os::InitializeMultiWaitHolder() で初期化が可能な型である必要があります。
    *
    * @pre
    *   - IsInitialized() == false
    *   - pSyncObject != nullptr;
    *   - pSyncObject が初期化状態である
    *
    * @post
    *   - IsInitialized() == true
    *
    * @detail
    *   イベントハンドラオブジェクトを初期化します。 @n
    *   また、与えられた同期オブジェクトを使用して内部の多重待ちオブジェクトを初期化します。
    */
    template<typename T>
    void Initialize(T* pSyncObject) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pSyncObject);
        NN_SDK_REQUIRES(!IsInitialized());
        nn::os::InitializeMultiWaitHolder(&m_Holder, pSyncObject);
        nn::os::SetMultiWaitHolderUserData(&m_Holder, reinterpret_cast<uintptr_t>(this));
        m_IsInitialized = true;
        m_IsLinked = false;
    }

    /**
    * @brief イベントハンドラオブジェクトをファイナライズします。
    *
    * @pre
    *   - IsInitialized() == true
    *   - IsRegistered() == false
    *
    * @post
    *   - IsInitialized() == false
    *
    * @detail
    *   イベントハンドラオブジェクトをファイナライズします。
    */
    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!IsRegistered());
        nn::os::FinalizeMultiWaitHolder(&m_Holder);
        m_IsInitialized = false;
        m_IsLinked = false;
    }

    /**
    * @brief イベントハンドラオブジェクトが初期化済かを返します。
    *
    * @return イベントハンドラオブジェクトが初期化済かどうか
    *
    * @detail
    *   イベントハンドラオブジェクトが初期化済かを返します。
    */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    /**
    * @brief ユーザ設定パラメータを取得します。
    *
    * @return ユーザ設定パラメータ
    *
    * @detail
    *   ユーザ設定パラメータを取得します。
    */
    uintptr_t GetUserArg() const NN_NOEXCEPT
    {
        return m_pUserArg;
    }

    /**
    * @brief ユーザ設定パラメータを設定します。
    *
    * @param[in]    arg     パラメータ
    *
    * @detail
    *   ユーザ設定パラメータを設定します。
    */
    void SetUserArg(uintptr_t arg) NN_NOEXCEPT
    {
        m_pUserArg = arg;
    }

    /**
    * @brief イベントハンドラオブジェクト内部の多重待ちオブジェクトが、多重待ちオブジェクトヘッダにリンク済かを返します。
    *
    * @return 多重待ちオブジェクトヘッダにリンク済か
    *
    * @detail
    *   イベントハンドラオブジェクト内部の多重待ちオブジェクトが、多重待ちオブジェクトヘッダにリンク済かを返します。
    */
    bool IsRegistered() const NN_NOEXCEPT
    {
        return m_IsLinked;
    }

protected: // 直接的には EventHandlerLooper のみが使用するが、doxygen 表示のため protected に

    //! @name 実装必須メソッド
    //! @{

    /**
    * @brief イベントがシグナルされた際に呼び出される、イベントハンドラメソッドの本体です。
    *
    * @detail
    *   イベントがシグナルされた際に呼び出される、イベントハンドラメソッドの本体です。 @n
    *   派生クラスで実装を与える必要があります。
    */
    virtual void HandleEvent() NN_NOEXCEPT = 0;

    //! @}

private: // EventHandlerLooper のみが使用
    /**
    * @brief 多重待ちオブジェクトからイベントハンドラオブジェクトを取り出します。
    *
    * @param[in]    pHolder 多重待ちオブジェクト
    *
    * @pre
    *   - pHolder != nullptr
    *
    * @pre
    *   - 多重待ちオブジェクトが初期化済のイベントハンドラオブジェクトの一部である
    *
    * @detail
    *   多重待ちオブジェクトからイベントハンドラオブジェクトを取り出します。 @n
    *   多重待ちオブジェクトが初期化済のイベントハンドラオブジェクトの一部であることが確実な状況でのみ使用してください。
    */
    static IEventHandler& ToEventHandler(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        auto& handler = *reinterpret_cast<IEventHandler*>(nn::os::GetMultiWaitHolderUserData(pHolder));
        NN_SDK_REQUIRES(handler.IsInitialized());
        return handler;
    }

private: // EventHandlerLooper のみが使用
    /**
    * @brief イベントハンドラオブジェクト内部の多重待ちオブジェクトを、多重待ちオブジェクトヘッダにリンクします。
    *
    * @param[in]    pMultiWait 多重待ちオブジェクトヘッダ
    *
    * @pre
    *   - IsInitialized() == true
    *   - IsRegistered() == false
    *   - pMultiWait != nullptr
    *   - pMultiWait が初期化状態である
    *
    * @post
    *   - IsRegistered() == true
    *
    * @detail
    *   イベントハンドラオブジェクト内部の多重待ちオブジェクトを、多重待ちオブジェクトヘッダにリンクします。 @n
    *   本 API はスレッドセーフではありません。
    *   本 API の動作中に @a multiWait やイベントハンドラオブジェクトに対する操作は行なわないで下さい。
    */
    void Link(nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(!IsRegistered());
        NN_SDK_REQUIRES_NOT_NULL(pMultiWait);
        nn::os::LinkMultiWaitHolder(pMultiWait, &m_Holder);
        m_IsLinked = true;
    }

    /**
    * @brief イベントハンドラオブジェクト内部の多重待ちオブジェクトを、多重待ちオブジェクトヘッダからアンリンクします。
    *
    * @pre
    *   - IsInitialized() == true
    *   - IsRegistered() == true
    *
    * @post
    *   - IsRegistered() == false
    *
    * @detail
    *   イベントハンドラオブジェクト内部の多重待ちオブジェクトを、多重待ちオブジェクトヘッダからアンリンクします。 @n
    *   本 API はスレッドセーフではありません。
    *   本 API の動作中にイベントハンドラオブジェクトに対する操作は行なわないで下さい。
    */
    void Unlink() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(IsRegistered());
        nn::os::UnlinkMultiWaitHolder(&m_Holder);
        m_IsLinked = false;
    }

private:
    nn::os::MultiWaitHolderType m_Holder;
    uintptr_t m_pUserArg{ 0 };
    bool m_IsInitialized{ false };
    bool m_IsLinked{ false };
};

//! @}

}} // nn::ddsf
