﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   安全なダウンキャストのための型情報を扱う機能に関する宣言
 */

#pragma once

#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/ddsf/detail/ddsf_Log.h>
#include <nn/ddsf/detail/ddsf_CastSafe.h>

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

 /**
 * @brief 安全なダウンキャストのための型情報を持たせることを宣言するマクロです。
 *
 * @detail
 *   安全なダウンキャストのための型情報を持たせることを宣言するマクロです。 @n
 *   このマクロは、 @ref nn::ddsf::ICastSafe を継承したクラスの宣言内でのみ使用できます。 @n
 *   記述方法の例については @ref nn::ddsf::ICastSafe のリファレンスを参照してください。
 */
#define NN_DDSF_CAST_SAFE_DECL

 /**
 * @brief 安全なダウンキャストのための型情報を定義するマクロです。
 *
 * @param[in]  className    クラス名。完全修飾名で使用してください。
 * @param[in]  base         基底クラス名。完全修飾名で使用してください。
 *
 * @detail
 *   安全なダウンキャストのための型情報を定義するマクロです。 @n
 *   このマクロは、必ずルート名前空間直下に記述して呼び出してください。 @n
 *   @ref NN_DDSF_CAST_SAFE_DECL を宣言した各クラスについて、ただ一つの定義を持つようにしてください。 @n
 *   記述方法の例については @ref nn::ddsf::ICastSafe のリファレンスを参照してください。
 */
#define NN_DDSF_CAST_SAFE_DEFINE(className, base)

#else

#define NN_DDSF_CAST_SAFE_DECL \
    public: \
        virtual const ::nn::ddsf::detail::TypeTag& GetTypeTag() const NN_NOEXCEPT NN_OVERRIDE { return NnDetailDdTypeTagInstance_; } \
        static const ::nn::ddsf::detail::TypeTag NnDetailDdTypeTagInstance_;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

#define NN_DDSF_CAST_SAFE_DEFINE(className, base) \
    const ::nn::ddsf::detail::TypeTag className::NnDetailDdTypeTagInstance_(#className, base::NnDetailDdTypeTagInstance_);

#else

// TypeTag コンストラクタ側でも className を渡されても Release ビルドでは使用しないようになっているが、
// マクロ側もこの実装分岐をしておくことでクラス名文字列が生成されないことを保証
#define NN_DDSF_CAST_SAFE_DEFINE(className, base) \
    const ::nn::ddsf::detail::TypeTag className::NnDetailDdTypeTagInstance_(base::NnDetailDdTypeTagInstance_);

#endif

#endif

namespace nn { namespace ddsf {

//! @name 安全なダウンキャスト機能関連 API
//! @{

/**
* @brief        RTTI を使用せず安全なキャストを行うメソッドを提供するインタフェースクラスです。
*
* @detail
*   RTTI が無効な環境でも使用可能な、安全なキャストを行うメソッドを提供するインタフェースクラスです。@n
*   このクラスを直接実体化することはできません。@n
*
*   このクラスを public 継承したクラスを基底とする派生クラスは、 @ref NN_DDSF_CAST_SAFE_DECL マクロや @ref NN_DDSF_CAST_SAFE_DEFINE マクロを使用して動的型の情報をクラスに与えることにより、
*   基底クラスからのダウンキャストに専用のメソッドを用いることでキャスト先の静的型とオブジェクトの動的型をチェックすることができます。 @n
*
*   例として ddsf::ISession の派生クラスの実装サンプルを以下に示します。 @n
*   @code
*       // subsystem_Session.h
*       #include <nn/ddsf/ddsf_DriverSubsystemFramework.h>
*       namespace nn { namespace subsystem {
*           class SubsystemSession final :
*               public nn::ddsf::ISession
*           {
*               NN_DDSF_CAST_SAFE_DECL; // このマクロをクラスの宣言内に配置してください。
*
*               // ...
*           };
*       }} // namespace nn::subsystem
*   @endcode
*
*   @code
*       // subsystem_Session.cpp
*       NN_DDSF_CAST_SAFE_DEFINE(nn::subsystem::SubsystemSession, nn::ddsf::ISession); // このマクロをサブシステムの定義側 (cpp) に配置してください。
*   @endcode
*/
class ICastSafe
{
public:
    //! @name 安全なダウンキャスト
    //! @{

    /**
    * @brief    このオブジェクトの動的型が T& に対して安全にキャスト可能であるかを判定します。
    *
    * @tparam   T     キャスト先の型
    *
    * @return   このオブジェクトの動的型が指定した型と一致しているか
    *
    * @detail
    *   このオブジェクトの動的型が T& に対して安全にキャスト可能であるかを判定します。 @n
    *   主に、基底クラスにアップキャストされている状態のオブジェクトに対してこのメソッドを呼び出し、派生クラスに安全に戻せるかどうかを判定するのに使用します。 @n
    *
    *   この機能を使ってクラス間が安全にキャスト可能であるためには以下の要件を満たす必要があります。
    *       - T は ICastSafe の派生である （コンパイル時判定）
    *          - T が ddsf::ISession か ddsf::IDevice か ddsf::IDriver の派生であれば満たす
    *       - T のクラス宣言で @ref NN_DDSF_CAST_SAFE_DECL されている（コンパイル時判定）
    *       - T について @ref NN_DDSF_CAST_SAFE_DEFINE(T, BaseOfT) されている（リンク時判定）
    *       - このメソッドが呼ばれているオブジェクトの実体の型が、 T と厳密に同じか、または @ref NN_DDSF_CAST_SAFE_DEFINE において T の派生型である（実行時判定）
    *
    *   @code
    *       class Base : public nn::ddsf::ICastSafe
    *       {
    *           NN_DDSF_CAST_SAFE_DECL:
    *       };
    *       NN_DETAIL_DDSF_CAST_SAFE_DEFINE_ROOT(Base);
    *
    *       class Derived : public Base
    *       {
    *           NN_DDSF_CAST_SAFE_DECL;
    *       };
    *       NN_DDSF_CAST_SAFE_DEFINE(Derived, Base);
    *
    *       class Derived2 : public Base
    *       {
    *           NN_DDSF_CAST_SAFE_DECL;
    *       };
    *       NN_DDSF_CAST_SAFE_DEFINE(Derived2, Base);
    *
    *       void F()
    *       {
    *           Base base;
    *           Derived derived;
    *
    *           Base* pBase= &base;
    *           NN_LOG("Can I cast pBase to Derived*? - %s\n", pBase->IsCastableTo<Derived>() ? "YES" : "NO"); // "NO"
    *           NN_LOG("Can I cast pBase to Derived2*? - %s\n", pBase->IsCastableTo<Derived2>() ? "YES" : "NO"); // "NO"
    *
    *           Derived* pDerived= &derived;
    *           NN_LOG("Can I cast pDerived to Base*? - %s\n", pDerived->IsCastableTo<Base>() ? "YES" : "NO"); // "YES"
    *           NN_LOG("Can I cast pDerived to Derived*? - %s\n", pDerived->IsCastableTo<Derived>() ? "YES" : "NO"); // "YES"
    *           NN_LOG("Can I cast pDerived to Derived2*? - %s\n", pDerived->IsCastableTo<Derived2>() ? "YES" : "NO"); // "NO"
    *
    *           Base* pDerivedAsBase = &derived;
    *           NN_LOG("Can I cast pDerivedAsBase to Base*? - %s\n", pDerivedAsBase->IsCastableTo<Base>() ? "YES" : "NO"); // "YES"
    *           NN_LOG("Can I cast pDerivedAsBase to Derived*? - %s\n", pDerivedAsBase->IsCastableTo<Derived>() ? "YES" : "NO"); // "YES"
    *           NN_LOG("Can I cast pDerivedAsBase to Derived2*? - %s\n", pDerivedAsBase->IsCastableTo<Derived2>() ? "YES" : "NO"); // "NO"
    *       }
    *   @endcode
    */
    template <typename T>
    bool IsCastableTo() const NN_NOEXCEPT
    {
        const auto& srcTypeTag = GetTypeTag();
        const auto& destTypeTag = T::NnDetailDdTypeTagInstance_;
        return srcTypeTag.DerivesFrom(destTypeTag);
    }

    /**
    * @brief    このオブジェクトを指定した型に安全にキャストします。
    *
    * @tparam   T     キャスト先の型
    *
    * @return   キャスト結果
    *
    * @pre
    *   - IsCastableTo<T>() == true
    *
    * @detail
    *   このオブジェクトを指定した型に安全にキャストします。 @n
    *   このメソッドは内部で @ref IsCastableTo<T>() が true であることをアサートでチェックしており、動的型が一致しない場合はアサートに失敗します。
    */
    template <typename T>
    T& SafeCastTo() NN_NOEXCEPT
    {
        AssertCastableTo<T>();
        return static_cast<T&>(*this);
    }

    /**
    * @brief    このオブジェクトを指定した型に安全にキャストします。
    *
    * @tparam   T     キャスト先の型
    *
    * @return   キャスト結果
    *
    * @pre
    *   - IsCastableTo<T>() == true
    *
    * @detail
    *   このオブジェクトを指定した型に安全にキャストします。 @n
    *   このメソッドは内部で @ref IsCastableTo<T>() が true であることをアサートでチェックしており、動的型が一致しない場合はアサートに失敗します。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    template <typename T>
    const T& SafeCastTo() const NN_NOEXCEPT
    {
        AssertCastableTo<T>();
        return static_cast<const T&>(*this);
    }

    /**
    * @brief    このオブジェクトへのポインタを指定した型へのポインタに安全にキャストします。
    *
    * @tparam   T     キャスト先の型
    *
    * @return   キャスト結果
    *
    * @pre
    *   - IsCastableTo<T>() == true
    *
    * @detail
    *   このオブジェクトを指定した型に安全にキャストします。 @n
    *   このメソッドは内部で @ref IsCastableTo<T>() が true であることをアサートでチェックしており、動的型が一致しない場合はアサートに失敗します。
    */
    template <typename T>
    T* SafeCastToPointer() NN_NOEXCEPT
    {
        AssertCastableTo<T>();
        return static_cast<T*>(this);
    }

    /**
    * @brief    このオブジェクトへのポインタを指定した型へのポインタに安全にキャストします。
    *
    * @tparam   T     キャスト先の型
    *
    * @return   キャスト結果
    *
    * @pre
    *   - IsCastableTo<T>() == true
    *
    * @detail
    *   このオブジェクトを指定した型に安全にキャストします。 @n
    *   このメソッドは内部で @ref IsCastableTo<T>() が true であることをアサートでチェックしており、動的型が一致しない場合はアサートに失敗します。 @n
    *   const メンバ関数内から呼び出すことができます。
    */
    template <typename T>
    const T* SafeCastToPointer() const NN_NOEXCEPT
    {
        AssertCastableTo<T>();
        return static_cast<const T*>(this);
    }

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    /**
    * @brief    このオブジェクトの動的型に対応するクラス名を取得します。
    *
    * @return   クラス名
    *
    * @detail
    *   このオブジェクトの動的型に対応するクラス名を取得します。 @n
    *   このメソッドは Release ビルドでは呼び出すことはできません。
    */
    const char* GetClassName() const NN_NOEXCEPT
    {
        return GetTypeTag().GetClassName();
    }
#endif

    //! @}

private:
    virtual const detail::TypeTag& GetTypeTag() const NN_NOEXCEPT = 0;

    template <typename T>
    inline void AssertCastableTo() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsCastableTo<T>(), "Object 0x%016llx (instantiated as %s) cannot be cast to %s.\n", this, GetClassName(), T::NnDetailDdTypeTagInstance_.GetClassName());
    }
};

//! @}

}} // namespace nn::ddsf
