﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief DHCP Client Public Data Types
 *
 * @details
 */

namespace nn { namespace bsdsocket { namespace dhcpc {

enum DhcpcConfig
{
    Config_MaxWhitelistEntries     = 16,
    Config_MaxBlacklistEntries     = 16,
    Config_MaxClientIdSize         = 48,
    Config_MaxHardwareAddressSize  = 6,
    Config_MaxIfNameSize           = 16,
    Config_MaxIf                   = 2,
    Config_MaxMtu                  = 1500,
    Config_MinMtu                  = 576,
    Config_MaxUserClassSize        = 255,
    Config_MaxVendorIdClassSize    = 255,
    Config_MaxVendorSize           = 255,
    Config_MaxHostNameSize         = 250,
    Config_MinBootpMessageSize     = 300,
    Config_MinLeaseTimeSeconds     = 20,
    Config_DefaultRenewalRatio     = 500, // T1
    Config_DefaultRebindRatio      = 875, // T2
    Config_InfiniteTime            = 0xffffffff,
    Config_MaxRouters              = 16,
};

/* DHCP Client Daemon Options */
enum ClientDaemonOption
{
    ClientDaemonOption_Arp              =   (1ULL << 0),
    ClientDaemonOption_Release          =   (1ULL << 1),
    ClientDaemonOption_Domain           =   (1ULL << 2),
    ClientDaemonOption_Gateway          =   (1ULL << 3),
    ClientDaemonOption_Static           =   (1ULL << 4),
    ClientDaemonOption_Debug            =   (1ULL << 5),
    ClientDaemonOption_LastLease        =   (1ULL << 7),
    ClientDaemonOption_Inform           =   (1ULL << 8),
    ClientDaemonOption_Request          =   (1ULL << 9),
    ClientDaemonOption_Ipv4ll           =   (1ULL << 10),
    ClientDaemonOption_Duid             =   (1ULL << 11),
    ClientDaemonOption_Persistent       =   (1ULL << 12),
    ClientDaemonOption_Daemonise        =   (1ULL << 14),
    ClientDaemonOption_Daemonised       =   (1ULL << 15),
    ClientDaemonOption_Test             =   (1ULL << 16),
    ClientDaemonOption_Master           =   (1ULL << 17),
    ClientDaemonOption_Hostname         =   (1ULL << 18),
    ClientDaemonOption_ClientId         =   (1ULL << 19),
    ClientDaemonOption_Link             =   (1ULL << 20),
    ClientDaemonOption_Quiet            =   (1ULL << 21),
    ClientDaemonOption_Background       =   (1ULL << 22),
    ClientDaemonOption_VendorRaw        =   (1ULL << 23),
    ClientDaemonOption_NowaitIp         =   (1ULL << 24), /* To force daemonise */
    ClientDaemonOption_WaitIp           =   (1ULL << 25),
    ClientDaemonOption_SlaacPrivate     =   (1ULL << 26),
    ClientDaemonOption_CsrWarned        =   (1ULL << 27),
    ClientDaemonOption_XidHwaddr        =   (1ULL << 28),
    ClientDaemonOption_Broadcast        =   (1ULL << 29),
    ClientDaemonOption_Dumplease        =   (1ULL << 30),
    ClientDaemonOption_Ipv6rs           =   (1ULL << 31),
    ClientDaemonOption_Ipv6raReqrdnss   =   (1ULL << 32),
    ClientDaemonOption_Ipv6raOwn        =   (1ULL << 33),
    ClientDaemonOption_Ipv6raOwnDefault =   (1ULL << 34),
    ClientDaemonOption_Ipv4             =   (1ULL << 35),
    ClientDaemonOption_Forked           =   (1ULL << 36),
    ClientDaemonOption_Ipv6             =   (1ULL << 37),
    ClientDaemonOption_Started          =   (1ULL << 38),
    ClientDaemonOption_Noalias          =   (1ULL << 39),
    ClientDaemonOption_IaForced         =   (1ULL << 40),
    ClientDaemonOption_Stopping         =   (1ULL << 41),
    ClientDaemonOption_Departed         =   (1ULL << 42),
    ClientDaemonOption_HostnameShort    =   (1ULL << 43),
    ClientDaemonOption_Exiting          =   (1ULL << 44),
    ClientDaemonOption_WaitIp4          =   (1ULL << 45),
    ClientDaemonOption_WaitIp6          =   (1ULL << 46),
    ClientDaemonOption_Dev              =   (1ULL << 47),
    ClientDaemonOption_Iaid             =   (1ULL << 48),
    ClientDaemonOption_Dhcp             =   (1ULL << 49),
    ClientDaemonOption_Dhcp6            =   (1ULL << 50),
    ClientDaemonOption_Nopfxdlg         =   (1ULL << 51),
    ClientDaemonOption_Pfxdlgonly       =   (1ULL << 52),
    ClientDaemonOption_Pfxdlgmix        =   (1ULL << 53),
    ClientDaemonOption_Ipv6raAutoconf   =   (1ULL << 54),
    ClientDaemonOption_AuthSend         =   (1ULL << 55),
    ClientDaemonOption_AuthRequire      =   (1ULL << 56),
    ClientDaemonOption_SendRequire      =   (1ULL << 55) | (1ULL << 56),
    ClientDaemonOption_RdmCounter       =   (1ULL << 57),
};

struct Lease
{
    bool                isValid;
    nn::socket::InAddr  addr;
    nn::socket::InAddr  subnetMask;
    nn::socket::InAddr  broadcastAddr;
    TimeSpan            leaseTime;
    TimeSpan            renewalTime;
    TimeSpan            rebindTime;
    nn::socket::InAddr  serverAddr;
    TimeSpan            leaseTimeStamp;
    uint32_t            cookie;
};

enum Indication
{
    Indication_InvalidLease = 0,
    Indication_NewLease,
    Indication_RenewedLease,
    Indication_ReboundLease,
    Indication_Failure
};

struct IndicationDataType
{
    Indication indication;
    char       ifName[Config_MaxIfNameSize];
    Result     failureCause;
    Lease      lease;
};

typedef void (*ClientCallbackFunctionType)(void *, IndicationDataType *);

struct InterfaceConfigType
{
    // Administrative callback
    ClientCallbackFunctionType callback;
    void                       *callbackContext;

    // Interface name
    char ifName[Config_MaxIfNameSize];

    // Interface address
    uint8_t hardwareAddress[Config_MaxHardwareAddressSize];

    // Protocol Options
    uint8_t         requestMask[32];
    uint8_t         requireMask[32];
    uint8_t         noMask[32];
    uint8_t         rejectMask[32];
    uint8_t         dstMask[32];

    // Daemon Options
    uint64_t        daemonOptions;

    // Vendor-Identifying Vendor Options
    uint32_t        vivoEnterprise;
    uint8_t         *pVivoData;
    uint8_t         vivoDataSize;

    // Parameters
    uint32_t        requestedAddr;
    uint32_t        serverAddr;
    uint8_t         userClass[Config_MaxUserClassSize + 2];
    uint8_t         vendorClassId[Config_MaxVendorIdClassSize + 2];
    uint8_t         vendor[Config_MaxVendorSize + 2];
    uint32_t        leaseTime;
    char            hostName[Config_MaxHostNameSize];
    DhcpProtFqdn    fqdn;
};

// Macros for manipulation of protocol option masks
#define DHCPC_ADD_PROT_OPTION_MASK(var, val) (var[val >> 3] |= 1 << (val & 7))
#define DHCPC_DEL_PROT_OPTION_MASK(var, val) (var[val >> 3] &= ~(1 << (val & 7)))
#define DHCPC_HAS_PROT_OPTION_MASK(var, val) (var[val >> 3] & (1 << (val & 7)))



} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn






