﻿using Nintendo.Nact.FileSystem;
using SigloNact.Utilities;
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Xunit;

namespace SigloNact.Test
{
    public class ConvertShowIncludesUtilTest
    {
        public ConvertShowIncludesUtilTest()
        {
        }

        [Fact]
        public void ParseMsvcShowIncludesTest()
        {
            string tempDir = Path.Combine(Environment.CurrentDirectory, nameof(ParseMsvcShowIncludesTest));
            Directory.CreateDirectory(tempDir);
            string testFilePath = Path.Combine(tempDir, "test.h");
            File.Create(testFilePath);

            string msvcShowIncludesFormat = "メモ: インクルード ファイル: ";
            string[] testTextLines = new string[] {
                // MSVC が 1 行目に出力するソースファイル名
                "test.cpp",
                // MSVC の showIncludes で出力される include 解析結果のフォーマット
                msvcShowIncludesFormat + testFilePath,
                // 無関係な文字列
                "warning",
                // 無関係なファイルパス
                @"C:\a\b.h",
                // include 解析結果のフォーマットにあわせた無関係な文字列
                msvcShowIncludesFormat + "warning",
                // include 解析結果のフォーマットにあわせた存在しないファイルパス
                msvcShowIncludesFormat + Path.Combine(tempDir, "noexistentpath.h"),
            };

            Assert.Equal(new string[] { testFilePath }, ConvertShowIncludesUtil.ParseMsvcShowIncludes(testTextLines).ToArray());
        }

        [Fact]
        public void GetFilteredPathsTest()
        {
            var testPaths = new FilePath[] {
                FilePath.CreateLocalFileSystemPath(@"C:\a\a.txt"),
                FilePath.CreateLocalFileSystemPath(@"C:\a\b\a.txt"),
                FilePath.CreateLocalFileSystemPath(@"C:\c\b.txt"),
            };

            // 指定したディレクトリ以下のファイルパスをフィルタ
            Assert.Equal(
                ConvertShowIncludesUtil.GetFilteredPaths(
                    testPaths,
                    new FilePath[] {
                        FilePath.CreateLocalFileSystemPath(@"C:\a\b"),
                    }
                ).ToArray(),
                new FilePath[] {
                    FilePath.CreateLocalFileSystemPath(@"C:\a\a.txt"),
                    FilePath.CreateLocalFileSystemPath(@"C:\c\b.txt"),
                });

            // 全てのファイルパスをフィルタ
            Assert.Equal(
                ConvertShowIncludesUtil.GetFilteredPaths(
                    testPaths,
                    new FilePath[] {
                        FilePath.CreateLocalFileSystemPath(@"C:\a"),
                        FilePath.CreateLocalFileSystemPath(@"C:\c"),
                    }
                ).ToArray(),
                new FilePath[] {
                });

            // フィルタ対象のパスが空
            Assert.Equal(
                ConvertShowIncludesUtil.GetFilteredPaths(
                    testPaths,
                    new FilePath[] {
                    }
                ).ToArray(),
                testPaths);

            // フィルタ対象が無関係なパス
            Assert.Equal(
                ConvertShowIncludesUtil.GetFilteredPaths(
                    testPaths,
                    new FilePath[] {
                        FilePath.CreateLocalFileSystemPath(@"D:\a"),
                        FilePath.CreateLocalFileSystemPath(@"C:\d"),
                    }
                ).ToArray(),
                testPaths);

            // フィルタ対象のパスの大文字小文字が異なる
            Assert.Equal(
                ConvertShowIncludesUtil.GetFilteredPaths(
                    testPaths,
                    new FilePath[] {
                        FilePath.CreateLocalFileSystemPath(@"C:\A\B"),
                    }
                ).ToArray(),
                new FilePath[] {
                    FilePath.CreateLocalFileSystemPath(@"C:\a\a.txt"),
                    FilePath.CreateLocalFileSystemPath(@"C:\c\b.txt"),
                });
        }
    }
}
