﻿using Nintendo.Nact;
using Nintendo.Nact.FileSystem;
using Nintendo.Nact.Testing;
using SigloNact.BuiltIns.ToolChain.GccClang;
using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Xunit;

namespace SigloNact.Test.BuiltIns.ToolChain.GccClang
{
    public class GccClangActoinsTest
    {
        public GccClangActoinsTest()
        {
            GlobalNact.Initialize();
        }

        [Fact]
        public void TestClangPreProcessCommandLine()
        {
            var workingDirectory = FilePath.CreateLocalFileSystemPath(@"c:\");
            var sourceDirectory = FilePath.CreateLocalFileSystemPath(@"c:\src");
            var outputDirectory = FilePath.CreateLocalFileSystemPath(@"c:\obj");
            var includePath0 = FilePath.CreateLocalFileSystemPath(@"c:\include");

            using (var context = CreateContext(workingDirectory, workingDirectory))
            {
                var helper = (NactActionHelperMock)context.Helper;
                var toolChainSpecifier = CreateClangSpecifier();

                helper.AddSimulatedProgramExecutionResult(0);

                var result = GccClangActions.GccClangPreprocessAction(
                    context,
                    toolChainSpecifier,
                    language: "C",
                    outputPreprocessedFile: outputDirectory.Combine("foo.i"),
                    sourceFile: sourceDirectory.Combine("foo.c"),
                    modelOptions: Array.Empty<string>(),
                    compileOptions: Array.Empty<string>(),
                    preprocessorMacros: new string[] { "FOO", "BAR=1" },
                    includeDirectories: new FilePath[] { includePath0 });

                var exec0 = result.ExecutedPrograms.ElementAt(0);
                var expected = new string[]
                {
                    @"""C:\clang\nx\armv7l\bin\clang.exe""",
                    "-x",
                    "c",
                    "-IC:\\include",
                    "-D",
                    "\"FOO\"",
                    "-D",
                    "\"BAR=1\"",
                    "-P",
                    "-E",
                    "-c",
                    @"C:\src\foo.c",
                    "-o",
                    @"C:\obj\foo.i"
                };

                Assert.Equal(expected, exec0.CommandLine.Split(' '));
            }
        }

        [Fact]
        public void TestParseClangBugReportPaths()
        {
            // by SIGLO-47868
            var dumpFilePath = @"D:\home\teamcity\temp\buildTmp\clang.exe-53b4c3.dmp";
            var intermediatesCppPath = @"D:\home\teamcity\temp\buildTmp\UnitTestDetailCommonNfpStoreDataRawExtention-615524.cpp";
            var intermediatesShPath = @"D:\home\teamcity\temp\buildTmp\UnitTestDetailCommonNfpStoreDataRawExtention-615524.sh";
            var clangBugReportMessage = new List<string>()
            {
                $@"Wrote crash dump file ""{dumpFilePath}""",
                @"0x00007FF68B9537B8(0x0000001A017DCC18 0x0000001A017DBF70 0x0000000000000000 0x0000001A7D8A21D0)",
                @"0x00007FF68B9536FE(0x0000001A017DBF58 0x00007FF689A80000 0x0000001A7D79AF60 0x0000001A017DB651)",
                @"0x00007FF68B1557E9(0x0000000000000000 0x0000001A00000000 0x0000001A7D79AF60 0x0000001A7EC66440)",
                @"0x00007FF68B155E81(0x0000001A017DCF78 0x00007FF689A80000 0x0000001A7D79AF60 0x0000001A7D79B180)",
                @"0x00007FF68B15456D(0x0000000000000000 0x0000001A7D79AF60 0x0000001A7D8A21D0 0x0000001A7EC66440)",
                @"0x00007FF68B155E81(0x0000001A017DD138 0x00007FF689A80000 0x0000001A7D79AF60 0x00007FF68B8AC800)",
                @"0x00007FF68B15456D(0x0000000000000000 0x0000001A7D79AF60 0x0000001A017DD040 0x00007FF68B203007)",
                @"0x00007FF68B155E81(0x0000001A7D79AF60 0x0000001A7D79AF60 0x0000001A017DD040 0x0000001A7D79B0E0)",
                @"0x00007FF68B155ABF(0x0000001A7D79AF60 0x0000000000000000 0x0000001A7EC66440 0x0000001A7EC66440)",
                @"0x00007FF68B13B19F(0x0000001A017DB651 0x0000001A017DBC80 0x0000001A017DBC80 0x0000000000000000)",
                @"0x00007FF68B13FCFF(0x0000000000000001 0x0000001A017DB651 0x0000001A017DB651 0x0000000000000001)",
                @"0x00007FF68B135A8C(0x0000001A017DD040 0x0000001A7D8A4AD0 0x0000001A7D8A4AD0 0x0000001A017DD040)",
                @"0x00007FF68AF08140(0x0000001A7D8A4AD0 0x0000000000000000 0x00007FF689A88823 0x000000000000001E)",
                @"0x00007FF68AF06D27(0x0000001A017DB140 0x0000001A7D79D4A0 0x0000001A7D8A4AD0 0x0000000000000003)",
                @"0x00007FF68AEED9DB(0x0000001A7D8A4AD0 0x0000001A7D8A4AD0 0x0000000000000000 0x0000000000000000)",
                @"0x00007FF68AEEDC39(0x0000000000000000 0x0000001A7D79D799 0x0000001A7D8A4AD0 0x0000001A7D79D870)",
                @"0x00007FF68AEEE5FC(0x0000001A7D810000 0x0000001A7D79D870 0x0000001A7D79D820 0x0000000000000000)",
                @"0x00007FF68AEF0D2B(0x0000001A7D86B680 0x0000001A7D86B680 0x0000000000000000 0x0000001A7D8A4AD0)",
                @"0x00007FF68AEE9E2E(0x0000001A7D7ED700 0x0000000000000000 0x0000001A7D7ED7F0 0x0000000000000000)",
                @"0x00007FF68AAF5109(0x0000001A7D7ED7F0 0x0000000000000000 0x0000000000000000 0x000000000000000F)",
                @"0x00007FF68BD97B12(0xFFFFFFFFFFFFFFFF 0x0000001A7D811750 0x0000000000000000 0x0000000000000000)",
                @"0x00007FF68AAF4FA5(0x0000001A7D806F60 0x0000001A7D7ED7F0 0x0000000000000000 0x0000001A7D806F60)",
                @"0x00007FF68AABCD70(0x0000001A7D814ED0 0x0000001A7D79DCF9 0x0000001A7D811750 0x0000000000000000)",
                @"0x00007FF68AB74314(0x0000001A7D7ED7F0 0x0000000000000000 0x0000001A7D7F4980 0x0000001A7D7ED7F0)",
                @"0x00007FF689B57231(0x00007C31C542042A 0x00007FF689B4F879 0x00007FF689AAAE87 0x000000007D79E600)",
                @"0x00007FF689B514E1(0x000000007D79E600 0x0000001A7D79E500 0x0000000000000000 0x0000000000000000)",
                @"0x00007FF689B542E7(0x00007FF689A97400 0x0000000000000000 0x00007FF688BEF000 0x0000000000000000)",
                @"0x00007FF68BB8D0CD(0x00007FF689A9742C 0x00007FF688BEF000 0x0000000000000000 0x0000000000000000)",
                @"0x00007FFE534613D2(0x00007FFE534613B0 0x0000000000000000 0x0000000000000000 0x0000000000000000), BaseThreadInitThunk() + 0x22 bytes(s)",
                @"0x00007FFE539154E4(0x0000000000000000 0x0000000000000000 0x0000000000000000 0x0000000000000000), RtlUserThreadStart() + 0x34 bytes(s)",
                @"clang.exe: error: clang frontend command failed due to signal(use -v to see invocation)",
                @"Rynda clang version 1.2.1(based on LLVM 3.9.1-9a98850)",
                @"Target: aarch64-nintendo-nx-elf",
                @"Thread model: posix",
                @"InstalledDir: D:\Tools\clang-for-nx.1.2.1\nx\aarch64\bin",
                @"clang.exe: note: diagnostic msg: PLEASE submit a bug report to the Toolchain for NX team and include the crash backtrace, preprocessed source, and associated run script.",
                @"clang.exe: note: diagnostic msg:",
                @"********************",
                @"",
                @"PLEASE ATTACH THE FOLLOWING FILES TO THE BUG REPORT:",
                @"Preprocessed source(s) and associated run script(s) are located at:",
                $@"clang.exe: note: diagnostic msg: {intermediatesCppPath}",
                $@"clang.exe: note: diagnostic msg: {intermediatesShPath}",
                @"clang.exe: note: diagnostic msg:",
                @"",
                @"",
                @"********************",
                @"------"
            };

            var toolChainSpecifier = CreateClangSpecifier();
            var clangCommandHelper = GccClangUtil.CreateCommandHelper(toolChainSpecifier);
            var actual = clangCommandHelper.GetBugReportFilesFromOutput(clangBugReportMessage).ToArray();
            var expected = new[] { dumpFilePath, intermediatesCppPath, intermediatesShPath };

            Assert.Equal(expected, actual);
        }

        private static NactActionContextMock CreateContext(
            FilePath workingDirectory,
            FilePath temporaryDirectory)
        {
            return new NactActionContextMock(
                c => new NactActionHelperMock(c),
                workingDirectory,
                temporaryDirectory);
        }

        private static GccClangToolChainSpecifier CreateClangSpecifier()
        {
            return new GccClangToolChainSpecifier(
                toolChainKind: "Clang",
                toolChainRoot: FilePath.CreateLocalFileSystemPath(@"c:\clang"),
                architecture: "arm",
                isPic: false);
        }
    }
}
