﻿using Nintendo.Nact.BuiltIn;
using Nintendo.Nact.Execution;
using Nintendo.Nact.FileSystem;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;

namespace SigloNact.BuiltIns.Bundle
{
    [NactActionFunctionContainer]
    public static class GenerateBundleContainer
    {
        [NactActionFunction]
        public static NactActionResult GenerateBundle(
            INactActionContext context,
            FilePath outputBundleFile,
            IEnumerable<FilePath> inputModuleFiles,
            IEnumerable<FilePath> dependencyModuleFiles,
            string type,
            FilePath rootPath)
        {
            Generate(
                outputBundleFile.PathString,
                inputModuleFiles.Select(x => x.PathString),
                dependencyModuleFiles.Select(x => x.PathString),
                type,
                rootPath.PathString);

            return NactActionResult.CreateSuccess(
                inputModuleFiles.Concat(dependencyModuleFiles).ToArray(),
                new[] { outputBundleFile });
        }

        private static void Generate(string outputBundleFile, IEnumerable<string> inputModuleFiles, IEnumerable<string> dependencyModuleFiles, string type, string rootPath)
        {
            string[] header =
            {
                "# このファイルは直接編集しないで下さい。",
                "# このファイルは自動生成されます。",
                "# Don't edit this file directly.",
                "# This file is generated automatically.",
                "Bundle:",
                string.Concat("  Type: ", type),
                "  Files:"
            };
            string[] files = inputModuleFiles.SelectMany(file => ParseModuleFile(file, rootPath)).Distinct().ToArray();
            Array.Sort(files, StringComparer.InvariantCulture);
            string[] filterTargetFiles = dependencyModuleFiles.SelectMany(file => ParseModuleFile(file, rootPath)).Distinct().ToArray();
            string[] fileLines = files.Except(filterTargetFiles).Select(x => string.Concat("    - ", x)).ToArray();
            string[] bundleLines = header.Concat(fileLines.Length > 0 ? fileLines : new string[] { "    []" }).ToArray();

            File.WriteAllLines(outputBundleFile, bundleLines, new UTF8Encoding(true));
        }

        private static string[] ParseModuleFile(string moduleFile, string rootPath)
        {
            // パス区切り文字はスラッシュに統一
            return File.ReadAllLines(moduleFile).Where(x => !string.IsNullOrEmpty(x)).Select(x => GetRootRelativePath(x.Replace('\\', '/'), rootPath.Replace('\\', '/'))).ToArray();
        }

        private static string GetRootRelativePath(string targetPath, string rootPath)
        {
            return targetPath.StartsWith(rootPath) ? targetPath.Substring(rootPath.Length + 1) : targetPath;
        }
    }
}
