﻿using System;
using System.Collections.Generic;
using System.Linq;

namespace SigloNact.BuiltIns.BuildSystem
{
    public class ResourceConversionRuleAggregator : TargetRuleAggregator<string[][], ResourceConversionRuleAggregator.TypedCommandLineAndDependency, string>
    {
        public override TypedCommandLineAndDependency Translate(string[][] input)
        {
            return new TypedCommandLineAndDependency(input);
        }

        public override string GetGroupKey(TypedCommandLineAndDependency data)
        {
            return data.CommandLine;
        }

        public override void Validate(TypedCommandLineAndDependency[] group)
        {
            TypedCommandLineAndDependency first = null;

            foreach (var other in group)
            {
                if (first == null)
                {
                    first = other;
                }
                else
                {
                    if (!other.HasEqualInputsWith(first))
                    {
                        throw new TargetRuleAggregatorException(string.Format(
                            "Although two command line strings are identical, different input files are specified.\n" +
                            "Command Line = {0}\n" +
                            "Inputs1      = [{1}]\n" +
                            "Inputs2      = [{2}]\n",
                            other.CommandLine, string.Join(", ", first.Inputs), string.Join(", ", other.Inputs)
                        ));
                    }
                    if (!other.HasEqualOutputsWith(first))
                    {
                        throw new TargetRuleAggregatorException(string.Format(
                            "Although two command line strings are identical, different output files are specified.\n" +
                            "Command Line = {0}\n" +
                            "Outputs1     = [{1}]\n" +
                            "Outputs2     = [{2}]\n",
                            other.CommandLine, string.Join(", ", first.Outputs), string.Join(", ", other.Outputs)
                        ));
                    }
                    if (!other.HasEqualRuntimeSourcesWith(first))
                    {
                        throw new TargetRuleAggregatorException(string.Format(
                            "Although two command line strings are identical, different runtime files are specified.\n" +
                            "Command Line    = {0}\n" +
                            "RuntimeSources1 = [{1}]\n" +
                            "RuntimeSources2 = [{2}]\n",
                            other.CommandLine, string.Join(", ", first.RuntimeSources), string.Join(", ", other.RuntimeSources)
                        ));
                    }
                }
            }
        }

        public override object Translate(TypedCommandLineAndDependency output)
        {
            // narl 側で使用するのはコマンドライン文字列だけ
            return output.CommandLine;
        }
        public class TypedCommandLineAndDependency
        {
            public string CommandLine { get; private set; }
            public IEnumerable<string> Inputs { get; private set; }
            public IEnumerable<string> Outputs { get; private set; }
            public IEnumerable<string> RuntimeSources { get; private set; }

            public TypedCommandLineAndDependency(string[][] input)
            {
                if (input.Length != 4)
                {
                    throw new TargetRuleAggregatorException("list length must be 4: CommandLine, Inputs, Outputs, and RuntimeSources");
                }

                CommandLine = input[0][0];
                Inputs = input[1];
                Outputs = input[2];
                RuntimeSources = input[3];
            }

            public bool HasEqualInputsWith(TypedCommandLineAndDependency tcld)
            {
                return Inputs.OrderBy(x => x).SequenceEqual(tcld.Inputs.OrderBy(x => x));
            }

            public bool HasEqualOutputsWith(TypedCommandLineAndDependency tcld)
            {
                return Outputs.OrderBy(x => x).SequenceEqual(tcld.Outputs.OrderBy(x => x));
            }

            public bool HasEqualRuntimeSourcesWith(TypedCommandLineAndDependency tcld)
            {
                return RuntimeSources.OrderBy(x => x).SequenceEqual(tcld.RuntimeSources.OrderBy(x => x));
            }
        }
    }
}
