﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.MakeVisualStudioProject
{
    public class SigloProjectUtility
    {
        public const string DefaultSpec = "Generic";
        public const string DefaultVsVersion = "VS2012";
        public const string DefaultToolset = "DefaultToolset";

        public static ProjectConfiguration CreateProjectConfiguration(
            string buildType, string platform, string spec = DefaultSpec, string vsVersion = DefaultVsVersion, string toolset = DefaultToolset)
        {
            return new ProjectConfiguration(GetConfiguration(buildType, spec, vsVersion, toolset), platform);
        }

        public static string GetBuildType(string configuration)
        {
            return ParseConfiguration(configuration).BuildType;
        }
        public static string GetBuildType(ProjectConfiguration pc)
        {
            return GetBuildType(pc.Configuration);
        }

        public static string GetSpec(string configuration)
        {
            return ParseConfiguration(configuration).Spec;
        }
        public static string GetSpec(ProjectConfiguration pc)
        {
            return GetSpec(pc.Configuration);
        }

        public static string GetVsVersion(string configuration)
        {
            return ParseConfiguration(configuration).VsVersion;
        }
        public static string GetVsVersion(ProjectConfiguration pc)
        {
            return GetVsVersion(pc.Configuration);
        }

        public static string GetToolset(string configuration)
        {
            return ParseConfiguration(configuration).Toolset;
        }
        public static string GetToolset(ProjectConfiguration pc)
        {
            return GetToolset(pc.Configuration);
        }

        public static string GetConfiguration(string buildType, string spec = DefaultSpec, string vsVersion = DefaultVsVersion, string toolset = DefaultToolset)
        {
            if (buildType == null)
            {
                throw new ArgumentNullException("buildType");
            }
            if (spec == null)
            {
                throw new ArgumentNullException("spec");
            }
            if (vsVersion == null)
            {
                throw new ArgumentNullException("vsVersion");
            }
            if (toolset == null)
            {
                throw new ArgumentNullException(nameof(toolset));
            }

            return string.Join("_",
                (spec == DefaultSpec ? Enumerable.Empty<string>() : new string[] { spec })
                .Concat(vsVersion == DefaultVsVersion ? Enumerable.Empty<string>() : new string[] { vsVersion })
                .Concat(toolset == DefaultToolset ? Enumerable.Empty<string>() : new string[] { toolset })
                .Concat(new string[] { buildType }));
        }

        private class ConfigurationParseResult
        {
            public string Spec;
            public string VsVersion;
            public string Toolset;
            public string BuildType;
        }

        private static ConfigurationParseResult ParseConfiguration(string configuration)
        {
            if (configuration == null)
            {
                throw new ArgumentNullException(nameof(configuration));
            }

            var split = configuration.Split(new[] { '_' });

            var buildType = split.Last();
            if (!IsBuildType(buildType))
            {
                throw new ArgumentException($"構成名 '{configuration}' が Siglo の Visual Studio プロジェクト仕様に沿っていません。ビルドタイプ名が無効です。");
            }

            var spec = DefaultSpec;
            var vsVersion = DefaultVsVersion;
            var toolset = DefaultToolset;

            switch (split.Length)
            {
                case 4:
                    // 0    1     2       3
                    // SPEC_VSVER_TOOLSET_BUILDTYPE
                    spec = split[0];
                    vsVersion = split[1];
                    toolset = split[2];
                    break;
                case 3:
                    if (IsVsVersion(split[0]))
                    {
                        // VSVER_TOOLSET_BUILDTYPE
                        vsVersion = split[0];
                        toolset = split[1];
                    }
                    else if (IsVsVersion(split[1]))
                    {
                        // SPEC_VSVER_BUILDTYPE
                        spec = split[0];
                        vsVersion = split[1];
                    }
                    else
                    {
                        // SPEC_TOOLSET_BUILDTYPE
                        spec = split[0];
                        toolset = split[1];
                    }
                    break;
                case 2:
                    if (IsVsVersion(split[0]))
                    {
                        // VSVER_BUILDTYPE
                        vsVersion = split[0];
                    }
                    else if (IsToolset(split[0]))
                    {
                        // TOOLSET_BUILDTYPE
                        toolset = split[0];
                    }
                    else
                    {
                        // SPEC_BUILDTYPE
                        spec = split[0];
                    }
                    break;
                case 1:
                    // BUILDTYPE
                    break;
                default:
                    System.Diagnostics.Debug.Assert(split.Length > 4);
                    throw new ArgumentException($"構成名 '{configuration}' が Siglo の Visual Studio プロジェクト仕様に沿っていません。アンダースコアの数が多すぎます。");
            }

            if (!IsVsVersion(vsVersion))
            {
                throw new ArgumentException($"構成名 '{configuration}' が Siglo の Visual Studio プロジェクト仕様に沿っていません。Visual Studio バージョンが無効です。");
            }
            if (!IsToolset(toolset))
            {
                throw new ArgumentException($"構成名 '{configuration}' が Siglo の Visual Studio プロジェクト仕様に沿っていません。ツールセット名が無効です。");
            }

            return new ConfigurationParseResult()
            {
                Spec = spec,
                VsVersion = vsVersion,
                Toolset = toolset,
                BuildType = buildType
            };
        }

        private static bool IsBuildType(string str)
        {
            return str == "Debug" || str == "Develop" || str == "Release";
        }

        private static bool IsVsVersion(string str)
        {
            return str.StartsWith("VS");
        }

        private static bool IsToolset(string str)
        {
            return str == DefaultToolset || str == "MSVC" || str == "ClangC2";
        }
    }
}
