﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Nintendo.MakePropertySheets.Constructor
{
    internal class ProjectElementConstructorDependencyResolver : IEnumerable<IEnumerable<ProjectElement>>
    {
        private IEnumerable<ProjectElement> m_Elements;

        public ProjectElementConstructorDependencyResolver(IEnumerable<ProjectElement> elements)
        {
            m_Elements = elements;
        }

        public IEnumerator<IEnumerable<ProjectElement>> GetEnumerator()
        {
            return new Enumerator(m_Elements);
        }

        private IEnumerator<IEnumerable<ProjectElement>> GetEnumerator_() => GetEnumerator();
        IEnumerator IEnumerable.GetEnumerator() => GetEnumerator_();

        private class Enumerator : IEnumerator<IEnumerable<ProjectElement>>
        {
            private class ProjectElementHolder
            {
                private static readonly Regex s_PropertyPattern = new Regex(@"(\$\([a-zA-Z_][0-9a-zA-Z_]*\))", RegexOptions.Compiled);
                private static readonly Regex s_MetadataPattern = new Regex(@"(%\(([a-zA-Z_][0-9a-zA-Z_\.]*)\))", RegexOptions.Compiled);

                public ProjectElement Element { get; }
                public string[] Products { get; }
                public string[] Dependencies { get; }

                public ProjectElementHolder(ProjectElement element)
                {
                    Element = element;
                    Products = EnumerateProducts(Element).ToArray();
                    Dependencies = EnumerateDependencies(Element).Except(Products).ToArray();  // 自分自身への依存は除く
                }

                private IEnumerable<string> EnumerateProducts(ProjectElement element)
                {
                    var products = new HashSet<string>();

                    if (element is Property.ProjectProperty)
                    {
                        var property = element as Property.ProjectProperty;

                        products.Add($"$({property.Name})");
                    }
                    else if (element is Property.ProjectPropertyGroup)
                    {
                        var propertyGroup = element as Property.ProjectPropertyGroup;

                        foreach (var property in propertyGroup.Properties)
                        {
                            products.Add($"$({property.Name})");
                        }
                    }
                    else if (element is ItemDefinition.ProjectMetadata)
                    {
                        throw new ImplementationErrorException($"{nameof(ItemDefinition.ProjectMetadata)} is not used as an element of {nameof(ItemDefinition.ProjectItemDefinition)}");
                    }
                    else if (element is ItemDefinition.ProjectItemDefinition)
                    {
                        var itemDefinition = element as ItemDefinition.ProjectItemDefinition;

                        foreach (var metadata in itemDefinition.Metadatas)
                        {
                            products.Add($"%({itemDefinition.ItemType}.{metadata.Name})");
                        }
                    }
                    else if (element is Target.ProjectTarget)
                    {
                        // TORIAEZU: products が空だと「何も出力しない」と見なされるので、適当な文字列を出力
                        var target = element as Target.ProjectTarget;

                        products.Add($"T({target.Name})");
                    }
                    else if (element is Target.ProjectTask)
                    {
                        throw new ImplementationErrorException($"{nameof(Target.ProjectTask)} is not used as an element of {nameof(Target.ProjectTarget)}");
                    }
                    else
                    {
                        throw new NotImplementedException();
                    }

                    return products.ToArray();
                }

                private IEnumerable<string> EnumerateDependencies(ProjectElement element)
                {
                    var dependencies = new HashSet<string>();

                    if (element is Property.ProjectProperty)
                    {
                        var property = element as Property.ProjectProperty;

                        FindAndAddDependentProperties(property.Condition, dependencies);
                        FindAndAddDependentProperties(property.Value, dependencies);
                    }
                    else if (element is Property.ProjectPropertyGroup)
                    {
                        var propertyGroup = element as Property.ProjectPropertyGroup;

                        FindAndAddDependentProperties(propertyGroup.Condition, dependencies);

                        foreach (var property in propertyGroup.Properties)
                        {
                            FindAndAddDependentProperties(property.Condition, dependencies);
                            FindAndAddDependentProperties(property.Value, dependencies);
                        }
                    }
                    else if (element is ItemDefinition.ProjectMetadata)
                    {
                        throw new ImplementationErrorException($"{nameof(ItemDefinition.ProjectMetadata)} is not used as an element of {nameof(ItemDefinition.ProjectItemDefinition)}");
                    }
                    else if (element is ItemDefinition.ProjectItemDefinition)
                    {
                        var itemDefinition = element as ItemDefinition.ProjectItemDefinition;

                        FindAndAddDependentProperties(itemDefinition.Condition, dependencies);

                        foreach (var metadata in itemDefinition.Metadatas)
                        {
                            FindAndAddDependentProperties(metadata.Condition, dependencies);
                            FindAndAddDependentProperties(metadata.Value, dependencies);

                            FindAndAddDependentMetadatas(metadata.Condition, dependencies, itemDefinition.ItemType);
                            FindAndAddDependentMetadatas(metadata.Value, dependencies, itemDefinition.ItemType);

                        }
                    }
                    else if (element is Target.ProjectTarget)
                    {
                        var target = element as Target.ProjectTarget;

                        FindAndAddDependentProperties(target.Condition, dependencies);

                        foreach (var task in target.Tasks)
                        {
                            FindAndAddDependentProperties(task.Condition, dependencies);

                            foreach (var parameter in task.Parameters.Values)
                            {
                                FindAndAddDependentProperties(parameter, dependencies);
                            }
                        }
                    }
                    else if (element is Target.ProjectTask)
                    {
                        throw new ImplementationErrorException($"{nameof(Target.ProjectTask)} is not used as an element of {nameof(Target.ProjectTarget)}");
                    }
                    else
                    {
                        throw new NotImplementedException();
                    }

                    return dependencies.ToArray();
                }

                private void FindAndAddDependentProperties(string unevaluatedValue, HashSet<string> dependencies)
                {
                    foreach (Match match in s_PropertyPattern.Matches(unevaluatedValue))
                    {
                        dependencies.Add(match.Value);
                    }
                }

                private void FindAndAddDependentMetadatas(string unevaluatedValue, HashSet<string> dependencies, string itemType = null)
                {
                    foreach (Match match in s_MetadataPattern.Matches(unevaluatedValue))
                    {
                        if (match.Value.IndexOf('.') >= 0)
                        {
                            // ItemType が入った形式 %(ItemType.Metadata) なので、そのまま追加
                            dependencies.Add(match.Value);
                        }
                        else
                        {
                            // ItemType が入っていない形式 %(Metadata) なので、自分で埋め込む必要がある
                            if (string.IsNullOrEmpty(itemType))
                            {
                                throw new ImplementationErrorException("unknown item type");
                            }
                            dependencies.Add($"%({itemType}.{match.Groups[2]})");
                        }
                    }
                }
            }

            private IEnumerable<ProjectElement> m_Elements;

            private HashSet<string> m_Produced;
            private List<ProjectElementHolder> m_Producible;
            private Dictionary<string, List<ProjectElementHolder>> m_Blocked;

            public Enumerator(IEnumerable<ProjectElement> elements)
            {
                m_Elements = elements;

                Reset();
            }

            public IEnumerable<ProjectElement> Current => m_Producible.Select(x => x.Element);

            object IEnumerator.Current => Current;

            public void Dispose()
            {
            }

            public bool MoveNext()
            {
                foreach (var holder in m_Producible)
                {
                    foreach (var product in holder.Products)
                    {
                        if (!m_Blocked.ContainsKey(product))
                        {
                            m_Produced.Add(product);
                        }
                    }
                }
                m_Producible.Clear();

                var blockedProducts = m_Blocked.Keys.ToArray();
                foreach (var blockedProduct in blockedProducts)
                {
                    var holders = m_Blocked[blockedProduct].ToArray();

                    foreach (var holder in holders)
                    {
                        if (holder.Dependencies.All(x => m_Produced.Contains(x) || !m_Blocked.ContainsKey(x)))
                        {
                            foreach (var product in holder.Products)
                            {
                                m_Blocked[product].Remove(holder);  // reference equality で平気？
                            }

                            m_Producible.Add(holder);
                        }
                    }

                    if (m_Blocked[blockedProduct].Count == 0)
                    {
                        m_Blocked.Remove(blockedProduct);
                    }
                }

                // 循環依存、自分自身に対する依存があったら実装誤り
                if (!m_Producible.Any() && m_Blocked.Any())
                {
                    throw new ImplementationErrorException("circular dependencies detected");
                }
                return m_Producible.Any();
            }

            public void Reset()
            {
                m_Produced = new HashSet<string>();
                m_Producible = new List<ProjectElementHolder>();
                m_Blocked = new Dictionary<string, List<ProjectElementHolder>>();

                foreach (var element in m_Elements)
                {
                    var holder = new ProjectElementHolder(element);

                    foreach (var product in holder.Products)
                    {
                        if (!m_Blocked.ContainsKey(product))
                        {
                            m_Blocked[product] = new List<ProjectElementHolder>();
                        }
                        m_Blocked[product].Add(holder);
                    }
                }
            }
        }
    }
}
