# coding: UTF-8

# Copyright (C)Nintendo All rights reserved.
#
# These coded instructions, statements, and computer programs contain proprietary
# information of Nintendo and/or its licensed developers and are protected by
# national and international copyright laws. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without the
# prior written consent of Nintendo.
#
# The content herein is highly confidential and should be handled accordingly.

import itertools
import os
import optparse
import sys

# 名前空間名を正規化して、必ず '::' で始まるようにする
def normalize_namespace(name):
    if name.startswith(':'):
        return name
    else:
        return "::" + name

#
# Options
#
script_dir = os.path.dirname(os.path.abspath(__file__))
c_header_template_path = os.path.join(script_dir, "CImplTemplate.h")
c_impl_template_path = os.path.join(script_dir, "CImplTemplate.cpp")

optionParser = optparse.OptionParser()
optionParser.add_option("-i", action="store", dest="input_path", help="Input source file path.")
optionParser.add_option("-o", action="store", dest="output_path", help="Output source file path.")
optionParser.add_option("-n", action="store", dest="namespace", help="C++ API namespace.")
optionParser.add_option("-I", action="append", dest="include_statements", help="Additional include file.")
optionParser.add_option("-B", action="append", dest="build_options", help="Build options for libclang.")
optionParser.add_option("--clang-root", action="store", dest="clang_root", help="Libclang root path.")
optionParser.add_option("--output-header", action="store_true", dest="output_header", help="Output C header file path.")
optionParser.add_option("--output-impl", action="store_true", dest="output_impl", help="Output C impl file path.")
optionParser.add_option("--output-convert-error", action="store_true", dest="output_convert_error", help="Output convert error as comment.")
optionParser.add_option("--output-template-function", action="store_true", dest="output_template_function", help="Output template function as comment.")

(options, args) = optionParser.parse_args()

input_path = options.input_path
output_path = options.output_path
namespace = normalize_namespace(options.namespace)
c_prefix = namespace.replace(':', '')

include_statements = options.include_statements or []
build_options = options.build_options or []

output_header = options.output_header
output_impl = options.output_impl
output_convert_error = options.output_convert_error
output_template_function = options.output_template_function

if output_header and output_impl:
    print('Error: cannot specify "--output-header" and "--output-impl" at the same time.')
    sys.exit(1)

# clang の python binding をインポートします。
sys.path.append(os.path.join(options.clang_root, 'src/bindings/python'))
from clang import cindex
cindex.Config.set_library_path(os.path.join(options.clang_root, 'bin'))

#
# Classes
#
class CppImplParser(object):
    """C++ ソースコードのパーサーです。C++ 実装から関数定義を取り出すのに使います。"""

    func_decls = []

    def __visit_func_decl(self, cursor, current_namespace):
        params = []
        for child in cursor.get_children():
            if child.kind == cindex.CursorKind.PARM_DECL:
                params.append({"name": child.spelling, "type": child.type.spelling})

        self.func_decls.append({
            "is_template": False,
            "namespace": current_namespace,
            "name": cursor.spelling,
            "type": cursor.result_type.spelling,
            "params": params,
            "raw_comment": cursor.raw_comment,
            "brief_comment": cursor.brief_comment})

    def __visit_function_template(self, cursor, current_namespace):
        params = []
        for child in cursor.get_children():
            if child.kind == cindex.CursorKind.PARM_DECL:
                params.append({"name": child.spelling, "type": child.type.spelling})

        self.func_decls.append({
            "is_template": True,
            "namespace": current_namespace,
            "name": cursor.spelling,
            "type": cursor.result_type.spelling,
            "params": params,
            "raw_comment": cursor.raw_comment,
            "brief_comment": cursor.brief_comment})

    def __visit_namespace(self, cursor, current_namespace):
        if cursor.kind == cindex.CursorKind.NAMESPACE:
            next_namespace = current_namespace + '::' + cursor.displayname
            for child in cursor.get_children():
                self.__visit_namespace(child, next_namespace)
        elif cursor.kind == cindex.CursorKind.FUNCTION_DECL:
            if current_namespace == namespace:
                self.__visit_func_decl(cursor, current_namespace)
        elif cursor.kind == cindex.CursorKind.FUNCTION_TEMPLATE:
            if current_namespace == namespace:
                self.__visit_function_template(cursor, current_namespace)

    def __visit_translation_unit(self, cursor):
        if cursor.kind == cindex.CursorKind.TRANSLATION_UNIT:
            for child in cursor.get_children():
                self.__visit_namespace(child, '')
        elif cursor.kind == cindex.CursorKind.NAMESPACE:
            for child in cursor.get_children():
                self.__visit_namespace(child, normalize_namespace(child.displayname))

    def parse(self, path, build_options):
        index = cindex.Index.create() # libclang の構文解析インタフェース
        translation_unit = index.parse(path, build_options)
        self.__visit_translation_unit(translation_unit.cursor)

    def get_func_decls(self):
        return self.func_decls

class FuncDeclConverter():
    """C++ の関数定義を C の関数定義に変換するクラスです。"""

    class TypeConvertError(Exception):
        """C++ から C への型変換に失敗"""

    def get_identifier(self, cpp_type_name):
        # const, struct, class などを取り除く
        keywords = ["const", "struct", "class"]
        l = filter(lambda s: s not in keywords, cpp_type_name.split(' '))
        return ' '.join(l).strip()

    def get_c_func_name_from_comment(self, comment):
        # NN_DETAIL_C_FUNCTION_NAME=<func_name>
        symbol = "NN_DETAIL_C_FUNCTION_NAME"
        for line in comment.splitlines():
            if symbol in line:
                after_symbol = line[line.find(symbol) + len(symbol):]
                func_name = after_symbol.strip("= \t\n")
                return func_name

    def remove_cpp_keyword(self, cpp_type_name):
        # class などを取り除く
        keywords = ["class"]
        l = filter(lambda s: s not in keywords, cpp_type_name.split(' '))
        return ' '.join(l).strip()

    def convert_type(self, cpp_type_name):
        type_name = self.remove_cpp_keyword(cpp_type_name)
        identifier = self.get_identifier(type_name)

        if identifier.startswith("nn::"):
            # nn 名前空間の型は、"::" を取り除いたものを C での名前とする
            return type_name.replace(':', '')
        elif '::' in type_name:
            # その他の名前空間上で定義された方はエラー
            raise TypeConvertError()
        else:
            # 名前空間上の型でなければ、同じ名前を C でも使う
            return type_name

    def convert(self, cpp_func_decls):
        c_func_decls = []
        for cpp in cpp_func_decls:
            if "NN_DETAIL_NO_C_IMPLEMENTATION" in cpp["raw_comment"]:
                continue

            c = {"cpp": cpp}
            if "NN_DETAIL_C_FUNCTION_NAME" in cpp["raw_comment"]:
                c["name"] = self.get_c_func_name_from_comment(cpp["raw_comment"])
            else:
                c["name"] = c_prefix + cpp["name"]

            c["params"] = []
            try:
                for param in cpp["params"]:
                    c["params"].append({"type": self.convert_type(param["type"]), "name": param["name"]})
                c["type"] = self.convert_type(cpp["type"])
                c["error"] = False
            except TypeConvertError:
                c["error"] = True

            c_func_decls.append(c)

        return c_func_decls

class CImplGenerator(object):
    """C の関数定義を元に、C 実装のソースコードを生成するクラスです。"""

    def __init__(self, c_func_decls):
        self.c_func_decls = c_func_decls

    def make_full_func_name(self, cpp_func_decl):
        return cpp_func_decl["namespace"] + '::' + cpp_func_decl["name"]

    def make_func_signature(self, c_func_decl):
        params = [' '.join([param["type"], param["name"]]) for param in c_func_decl["params"]]
        return "{0} {1}({2})".format(c_func_decl["type"], c_func_decl["name"], ", ".join(params))

    def make_func_declaration(self, cpp_func_decl, c_func_decl):
        src = []
        if c_func_decl["error"]:
            src.append("// Failed to convert {0}".format(c_func_decl["name"]))
        else:
            before_func = ""
            if cpp_func_decl["is_template"]:
                before_func = "/* template function\n"

            after_func = ""
            if cpp_func_decl["is_template"]:
                after_func = "\n*/"

            return f"""
/**
 * @brief {cpp_func_decl["brief_comment"]}
 * @details 詳細は {self.make_full_func_name(cpp_func_decl)}() を参照してください。
 */
{before_func}NN_EXTERN_C {self.make_func_signature(c_func_decl)};{after_func}
"""

    def make_header(self):
        func_decls = []
        for c in self.c_func_decls:
            if c["error"] and not output_convert_error:
                continue
            if c["cpp"]["is_template"] and not output_template_function:
                continue
            func_decls.append(self.make_func_declaration(c["cpp"], c))

        return """
{0}

// C Linkage
#ifndef NN_EXTERN_C
#ifdef  __cplusplus
#define NN_EXTERN_C extern "C"
#else
#define NN_EXTERN_C extern
#endif
#endif

{1}
""".format("\n".join(include_statements), "".join(func_decls))

    def make_func_impl(self, cpp_func_decl, c_func_decl):
        if c_func_decl["error"]:
            return "// Failed to convert {0}".format(c_func_decl["name"])
        else:
            params = []
            for cpp_param, c_param in zip(cpp_func_decl["params"], c_func_decl["params"]):
                if cpp_param["type"] == c_param["type"]:
                    params.append("{0}".format(c_param["name"]))
                elif c_param["type"].endswith("*"):
                    params.append("reinterpret_cast<{0}>({1})".format(cpp_param["type"], c_param["name"]))
                else:
                    params.append("static_cast<{0}>({1})".format(cpp_param["type"], c_param["name"]))

            func_call_left = ""
            if c_func_decl["type"] != "void":
                func_call_left = "auto result = "

            return_statement = ""
            if c_func_decl["type"] == "nnResult":
                return_statement = "return nn::result::ConvertToC(result);"
            elif c_func_decl["type"] == "void":
                return_statement = "return;"
            else:
                return_statement = "return result;"

            before_func = ""
            if cpp_func_decl["is_template"]:
                before_func = "/* template function\n"

            after_func = ""
            if cpp_func_decl["is_template"]:
                after_func = "\n*/"

            return f"""
{before_func}{self.make_func_signature(c_func_decl)}
{{
    {func_call_left}{self.make_full_func_name(cpp_func_decl)}
    (
        {', '.join(params)}
    );
    {return_statement}
}}{after_func}
"""

    def make_impl(self):
        func_impls = []
        for c in self.c_func_decls:
            if c["error"] and not output_convert_error:
                continue
            if c["cpp"]["is_template"] and not output_template_function:
                continue
            func_impls.append(self.make_func_impl(c["cpp"], c))

        return """
{0}

extern "C"
{{

{1}

}}
""".format("\n".join(include_statements), "".join(func_impls))

#
# Entry point
#
cpp_parser = CppImplParser()

try:
    cpp_parser.parse(input_path, build_options)
except:
    print("Failed to process '{0}'.".format(input_path))
    print("Try compiling '{0}' and remove all compile errors.".format(input_path))
    print("Libclang will not work if its input is not correct C++ code.")
    sys.exit(1)

cpp_func_decls = cpp_parser.get_func_decls()

converter = FuncDeclConverter()
c_func_decls = converter.convert(cpp_func_decls)

c_generator = CImplGenerator(c_func_decls)
c_header = c_generator.make_header()
c_impl = c_generator.make_impl()

if output_header:
    with open(c_header_template_path, 'r', encoding='utf-8') as template:
        with open(output_path, 'w', encoding='utf-8') as out:
            out.write(template.read())
            out.write(c_header)

if output_impl:
    with open(c_impl_template_path, 'r', encoding='utf-8') as template:
        with open(output_path, 'w', encoding='utf-8') as out:
            out.write(template.read())
            out.write(c_impl)
