﻿using Microsoft.VisualStudio.Setup.Configuration;
using Microsoft.Win32;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LocateVSIXInstaller
{
    // - https://developercommunity.visualstudio.com/content/problem/26008/vs-2017-rtm-vsixinstaller-cannot-find-setup-engine.html
    //   15.2 より前のバージョンの Visual Studio 2017 の VSIXInstaller には
    //  「Visual Studio 2017 (バージョン 15.1 以前) と Visual Studio 2015 以前のバージョンの Visual Studio が
    //   両方インストールされている場合に両方をサポートする VS 拡張のインストールに失敗する」不具合がある。
    //
    //   NintendoSDK は 15.2 より前の VS2017 をサポートしないので、
    //   15.2 より前の VS2017 から得られた VSIXInstaller を LocateVSIXInstaller が返さないようにすることで回避する。
    //
    // - Microsoft.VisualStudio.Setup.Configuration.Interop 相互運用型は埋め込まれています。
    //
    /// <summary>
    /// VS2017 (15.2 以降), VS2015, VS2013 のうち最も新しい VSIXInstaller.exe のパスを標準出力に出力する。
    /// </summary>
    internal class Program
    {
        private const string VSIXInstallerFileName = "VSIXInstaller.exe";
        private const int REGDB_E_CLASSNOTREG = unchecked((int)0x80040154);
        private const int Vs2017MicroVersion_15_2 = 26430;

        public static int Main(string[] args)
        {
            var path =
                GetFromSetupConfiguration()
                ?? GetFromRegistry("14.0")
                ?? GetFromRegistry("12.0");

            if (path != null)
            {
                Console.Write(path);
                return 0;
            }
            else
            {
                Console.Error.WriteLine("Could not find VSIXInstaller for Visual Studio 2017 (15.2 or later), 2015 or 2013.");
                return 1;
            }
        }

        /// <summary>
        /// レジストリから VSIXInstaller.exe のパスを取得する
        /// </summary>
        /// <param name="vsVersion">Visual Studio のバージョン。 14.0 / 12.0</param>
        /// <returns>VSIXInstaller.exe のパス</returns>
        private static string GetFromRegistry(string vsVersion)
        {
            var vsInstallDir = RegistryUtil.GetSoftwareValue($@"Microsoft\VisualStudio\{vsVersion}", "InstallDir") as string;
            if (vsInstallDir == null)
            {
                return null;
            }

            return Path.Combine(vsInstallDir, VSIXInstallerFileName);
        }

        /// <summary>
        /// Setup Configuration API から VSIInstaller.exe のパスを取得する
        /// </summary>
        /// <returns>VSIXInstaller.exe のパス</returns>
        private static string GetFromSetupConfiguration()
        {
            var instances = GetApplicableInstancesWithVersion();

            // 最も新しいバージョンのインスタンスを採用する
            foreach (var e in instances.OrderByDescending(x => x.MajorMicroVersion))
            {
                var instance = e.Instance;

                // Visual Studio Installer (instance.GetEnginePath()) の VSIXInstaller は更新されない可能性があるので、
                // いずれかの製品の VSIXInstaller を使用する。
                var installationPath = instance.GetInstallationPath();
                var candidate = Path.Combine(installationPath, "Common7", "IDE", "VSIXInstaller.exe");
                if (File.Exists(candidate))
                {
                    return candidate;
                }
            }

            return null;
        }

        private static List<InstanceWithVersion> GetApplicableInstancesWithVersion()
        {
            var result = new List<InstanceWithVersion>();

            foreach (var e in EnumerateSetupInstances())
            {
                // Build Tools には Visual Studio 拡張をインストールできない。
                // Visual Studio 2017 がインストールされておらず BuildTools for Visual Studio 2017 だけがインストールされている場合は、
                // VS2015 以前の VSIXInstaller を使用する。
                if (e.GetProduct().GetId() == "Microsoft.VisualStudio.Product.BuildTools")
                {
                    continue;
                }

                Version versionNumber;
                if (!Version.TryParse(e.GetInstallationVersion(), out versionNumber))
                {
                    continue;
                }

                // 15.2 より前の VS2017 から得られた VSIXInstaller を使用しない
                var majorVersion = versionNumber.Major;
                var microVersion = versionNumber.Build;
                if (majorVersion == 15 && microVersion < Vs2017MicroVersion_15_2)
                {
                    continue;
                }

                result.Add(new InstanceWithVersion(e, majorVersion, microVersion));
            }

            return result;
        }

        private static IEnumerable<ISetupInstance2> EnumerateSetupInstances()
        {
            IEnumSetupInstances enumerator;

            try
            {
                enumerator = new SetupConfiguration().EnumAllInstances();
            }
            catch (System.Runtime.InteropServices.COMException ex) when (ex.HResult == REGDB_E_CLASSNOTREG)
            {
                // Visual Studio Installer がインストールされていない
                yield break;
            }

            var instance = new ISetupInstance2[1];

            while (true)
            {
                int fetched;
                enumerator.Next(instance.Length, instance, out fetched);

                if (fetched <= 0)
                {
                    yield break;
                }

                yield return instance[0];
            }
        }

        private struct InstanceWithVersion
        {
            public ISetupInstance2 Instance { get; }
            public Tuple<int, int> MajorMicroVersion { get; }

            public InstanceWithVersion(ISetupInstance2 instance, int majorVersion, int microVersion)
            {
                this.Instance = instance;
                this.MajorMicroVersion = Tuple.Create(majorVersion, microVersion);
            }
        }
    }
}
