﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace GitExternalRepository.Repository
{
    /// <summary>
    /// リポジトリ操作クラスの生成を行うクラスです。
    /// </summary>
    public class RepositoryFactory
    {
        /// <summary>
        /// リポジトリ操作クラスについての情報を表現するクラスです。
        /// </summary>
        private class ClassInformation
        {
            /// <summary>
            /// 型情報です。
            /// </summary>
            public Type Type { get; set; }

            /// <summary>
            /// リポジトリの属性情報です。
            /// </summary>
            public RepositoryDefinitionAttribute Attirubte { get; set; }
        }

        private Dictionary<RepositoryType, ClassInformation> Repositories { get; set; }

        public RepositoryFactory()
        {
            this.Repositories = this.CreateClassInformations();
        }

        /// <summary>
        /// クラス情報を生成します。
        /// </summary>
        /// <returns>クラス情報</returns>
        private Dictionary<RepositoryType, ClassInformation> CreateClassInformations()
        {
            var commands = new Dictionary<RepositoryType, ClassInformation>();

            foreach (var type in Utils.GetInheritedClasses<RepositoryBase>())
            {
                var attr = (RepositoryDefinitionAttribute)Attribute.GetCustomAttribute(type, typeof(RepositoryDefinitionAttribute));

                if (attr == null)
                {
                    throw new RepositoryDefinitionAttributeNotDefined(string.Format("'{0}' inherited RepositoryBase class is not defined the attribute 'RepositoryDefinitionAttribute'.", type.Name));
                }

                var command = new ClassInformation()
                {
                    Type = type,
                    Attirubte = attr,
                };

                commands.Add(attr.RepositoryType, command);
            }

            return commands;
        }

        /// <summary>
        /// 指定の名前のコマンドが定義されているかどうかを返します。
        /// </summary>
        /// <param name="repositoryType">コマンド名</param>
        /// <returns>定義されているなら True, そうでなければ False</returns>
        public bool Defines(RepositoryType repositoryType)
        {
            return this.Repositories.ContainsKey(repositoryType);
        }

        /// <summary>
        /// 指定の名前を持つコマンドのインスタンスを生成します。
        /// </summary>
        /// <param name="repositoryType">コマンド名</param>
        /// <returns>コマンドのインスタンス</returns>
        public RepositoryBase Create(RepositoryType repositoryType, params object[] args)
        {
            if (!this.Defines(repositoryType))
            {
                throw new RepositoryNotDefinedException(string.Format("The repository type '{0}' is not defined.", repositoryType));
            }

            return (RepositoryBase)System.Activator.CreateInstance(this.Repositories[repositoryType].Type, args);
        }
    }
}
