﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Repository;
using GitExternalRepository.Repository.Git;

namespace GitExternalRepository
{
    /// <summary>
    /// ツールの実行環境を表現します。
    /// </summary>
    public static class Environments
    {
        /// <summary>
        /// リポジトリリストのパスを返します。
        /// </summary>
        /// <returns>リポジトリリストパス</returns>
        public static string GetRepositoryListPath()
        {
            return PathUtility.Combine(Environments.GitRepository.GetRepositoryRoot(), ".exrepomodules");
        }

        /// <summary>
        /// アクティブリポジトリリストのパスを返します。
        /// </summary>
        /// <returns>アクティブリポジトリリストパス</returns>
        public static string GetActiveRepositoryListPath()
        {
            var path = PathUtility.Combine(Environments.GitRepository.GetMetaDirectory(), "config");
            if (!File.Exists(path))
            {
                using (var fs = File.Create(path)) { }
            }

            return path;
        }

        /// <summary>
        /// サブモジュールのリポジトリリストのパスを返します。
        /// </summary>
        /// <returns>リポジトリリストパス</returns>
        public static string GetSubmoduleRepositoryListPath()
        {
            return PathUtility.Combine(Environments.GitRepository.GetRepositoryRoot(), ".gitmodules");
        }

        /// <summary>
        /// .git 内のツールディレクトリのパスを取得します。
        /// </summary>
        /// <returns></returns>
        public static string GetApplicationDirectoryPath()
        {
            var path = PathUtility.Combine(Environments.GitRepository.GetMetaDirectory(), "exrepo");
            if (!Directory.Exists(path))
            {
                Directory.CreateDirectory(path);
            }

            return path;
        }

        /// <summary>
        /// 外部リポジトリの管理ディレクトリを保存するディレクトリのパスを取得します。
        /// </summary>
        /// <returns></returns>
        public static string GetModuleDirectoryPath()
        {
            var path = PathUtility.Combine(GetApplicationDirectoryPath(), "modules");
            if (!Directory.Exists(path))
            {
                Directory.CreateDirectory(path);
            }

            return path;
        }

        #region GitRepository シングルトン

        private static GitRepositoryCommand gitRepositoryCommand;

        /// <summary>
        /// <see cref="GitRepositoryBase"/> のインスタンスを返します。
        /// </summary>
        public static GitRepositoryBase GitRepository
        {
            get
            {
                // コマンド版を返します。
                return GitRepositoryCommand;
            }
        }

        /// <summary>
        /// <see cref="GitRepositoryCommand"/> のインスタンスを返します。
        /// </summary>
        public static GitRepositoryCommand GitRepositoryCommand
        {
            get
            {
                if (gitRepositoryCommand == null)
                {
                    gitRepositoryCommand = CreateGitRepositoryCommand();
                }

                return gitRepositoryCommand;
            }
        }

        /// <summary>
        /// <see cref="GitRepositoryCommand"/> のインスタンスを作成します。
        /// </summary>
        /// <returns>インスタンス</returns>
        private static GitRepositoryCommand CreateGitRepositoryCommand()
        {
            try
            {
                var gitRepository = new GitRepositoryCommand(".");

                if (!gitRepository.IsInsideRepository())
                {
                    throw new GitExternalRepositoryException("The current directory is not part of a git repository.");
                }

                gitRepository.Open();

                return gitRepository;
            }
            catch (GitRepositoryCommandNotFoundException e)
            {
                throw new GitExternalRepositoryException("Not found the command 'git'.", e);
            }
        }

        #endregion

        #region RepositoryFactory シングルトン

        private static RepositoryFactory repositoryFactory;

        /// <summary>
        /// <see cref="RepositoryFactory"/> のインスタンスを返します。
        /// </summary>
        public static RepositoryFactory RepositoryFactory
        {
            get
            {
                if (repositoryFactory == null)
                {
                    repositoryFactory = new RepositoryFactory();
                }

                return repositoryFactory;
            }
        }

        #endregion

        #region RepositoryList シングルトン

        private static RepositoryListGitConfig repositoryListGitConfig;

        /// <summary>
        /// <see cref="RepositoryListGitConfig"/> のインスタンスを返します。
        /// </summary>
        public static RepositoryListGitConfig RepositoryList
        {
            get
            {
                return RepositoryListGitConfig;
            }
        }

        /// <summary>
        /// <see cref="RepositoryListGitConfig"/> のインスタンスを返します。
        /// </summary>
        public static RepositoryListGitConfig RepositoryListGitConfig
        {
            get
            {
                if (repositoryListGitConfig == null)
                {
                    repositoryListGitConfig = new RepositoryListGitConfig(Environments.GetRepositoryListPath());
                }

                return repositoryListGitConfig;
            }
        }

        #endregion

        #region ActiveRepositoryList シングルトン

        private static ActiveRepositoryListGitConfig activeRepositoryListGitConfig;

        /// <summary>
        /// <see cref="ActiveRepositoryListGitConfig"/> のインスタンスを返します。
        /// </summary>
        public static ActiveRepositoryListGitConfig ActiveRepositoryList
        {
            get
            {
                return ActiveRepositoryListGitConfig;
            }
        }

        /// <summary>
        /// <see cref="RepositoryListGitConfig"/> のインスタンスを返します。
        /// </summary>
        public static ActiveRepositoryListGitConfig ActiveRepositoryListGitConfig
        {
            get
            {
                if (activeRepositoryListGitConfig == null)
                {
                    activeRepositoryListGitConfig = new ActiveRepositoryListGitConfig(Environments.GetActiveRepositoryListPath());
                }

                return activeRepositoryListGitConfig;
            }
        }

        #endregion

        #region SubmoduleRepositoryList シングルトン

        private static SubmoduleRepositoryListGitConfig submoduleRepositoryListGitConfig;

        /// <summary>
        /// <see cref="SubmoduleRepositoryListGitConfig"/> のインスタンスを返します。
        /// </summary>
        public static SubmoduleRepositoryListGitConfig SubmoduleRepositoryList
        {
            get
            {
                return SubmoduleRepositoryListGitConfig;
            }
        }

        /// <summary>
        /// <see cref="SubmoduleRepositoryListGitConfig"/> のインスタンスを返します。
        /// </summary>
        public static SubmoduleRepositoryListGitConfig SubmoduleRepositoryListGitConfig
        {
            get
            {
                if (submoduleRepositoryListGitConfig == null)
                {
                    submoduleRepositoryListGitConfig = new SubmoduleRepositoryListGitConfig(Environments.GetSubmoduleRepositoryListPath());
                }

                return submoduleRepositoryListGitConfig;
            }
        }

        #endregion
    }
}
