﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Repository;
using GitExternalRepository.Repository.Git;
using System.Reflection;

namespace GitExternalRepository.Configs
{
    public enum SubSectionType
    {
        None,
        Required,
    }

    /// <summary>
    /// git config ファイルを設定ファイルとして扱うクラスです。
    /// </summary>
    public class EntryListGitConfig<TEntry> where TEntry : new()
    {
        /// <summary>
        /// 設定ファイルパス
        /// </summary>
        public string FilePath { get; private set; }

        /// <summary>
        /// 扱うセクション名
        /// </summary>
        public string Section { get; private set; }

        /// <summary>
        /// 扱うサブセクション名
        /// </summary>
        public string SubSection { get; private set; }

        /// <summary>
        /// サブセクションの要不要を示す
        /// </summary>
        public SubSectionType SubSectionType { get; private set; }

        /// <summary>
        /// git config を操作するクラス
        /// </summary>
        private GitConfigCommand GitConfig { get; set; }

        /// <summary>
        /// 指定の git config ファイルを設定ファイルとして扱います。
        /// </summary>
        /// <param name="filepath">コンフィグファイルへのパス</param>
        /// <param name="section">扱うセクション名を指定します。</param>
        /// <param name="section">サブセクションの要不要を示します。</param>
        /// <param name="subSection">扱うサブセクション名を指定します。</param>
        public EntryListGitConfig(string filepath, string section, SubSectionType subSectionType = SubSectionType.None, string subSection = null)
        {
            this.FilePath = filepath;
            this.Section = section;
            this.SubSection = subSection;
            this.SubSectionType = subSectionType;
            this.GitConfig = new GitConfigCommand(filepath);
        }

        /// <summary>
        /// 設定ファイルを作成します。
        /// </summary>
        public void Create()
        {
            using (var fs = File.Create(this.GitConfig.FilePath)) { }
        }

        /// <summary>
        /// 設定ファイルの有無を確認します。
        /// </summary>
        /// <returns></returns>
        public bool Exists()
        {
            return File.Exists(this.GitConfig.FilePath);
        }

        /// <summary>
        /// エントリを追加します。
        /// </summary>
        /// <param name="entry">追加するエントリ</param>
        public void Add(TEntry entry)
        {
            var configSection = ConvertToGitConfigSection(entry);
            var sectionKey = CreateSectionKey(configSection.Section, configSection.SubSection);

            foreach (var value in configSection.Entries)
            {
                this.GitConfig.Set(sectionKey + "." + value.Key, value.Value);
            }
        }

        /// <summary>
        /// エントリを削除します。
        /// </summary>
        /// <param name="entry">削除するエントリ</param>
        public void Remove(TEntry entry)
        {
            var entrySection = GetSection(entry);
            var entrySubSection = GetSubSection(entry);
            var sectionKey = CreateSectionKey(entrySection, entrySubSection);

            this.GitConfig.RemoveSection(sectionKey);
        }

        /// <summary>
        /// リポジトリリストからエントリを読み込みます。
        /// </summary>
        /// <returns>リポジトリリストのエントリ</returns>
        public IEnumerable<TEntry> Read()
        {
            var sections = this.GitConfig.ListSection().Where(x =>
                {
                    // セクション名の一致するものを探します。
                    var result = (this.Section == x.Section);

                    switch (this.SubSectionType)
                    {
                        case Configs.SubSectionType.None:
                            {
                                // サブセクションを持たないものを探します。
                                result &= (x.SubSection == null);
                            }
                            break;
                        case Configs.SubSectionType.Required:
                            {
                                // サブセクションを持つものを探します。
                                result &= (x.SubSection != null);

                                if (this.SubSection != null)
                                {
                                    // サブセクションに指定がある場合は、これと一致するものを探します。
                                    result &= (this.SubSection == x.SubSection);
                                }
                            }
                            break;
                        default:
                            throw new ArgumentOutOfRangeException();
                    }

                    return result;
                });

            return sections.Select(x => ConvertToEntry(x));
        }

        /// <summary>
        /// 指定のセクション、サブセクションを持つセクション指定文字列を生成します。
        /// 扱うセクション、サブセクションが制限されている場合、この制限に適合しないセクション名、サブセクション名を
        /// 引数に与えている場合、本関数は失敗します。
        /// </summary>
        /// <param name="secion">セクション名を指定します。</param>
        /// <param name="subSection">サブセクション名を指定します。</param>
        /// <returns></returns>
        private string CreateSectionKey(string secion, string subSection)
        {
            var key = string.Empty;

            if (secion != null && this.Section != null)
            {
                if (secion != this.Section)
                {
                    throw new Exception("Section name should be same.");
                }
            }
            else if (this.Section != null)
            {
                key += this.Section;
            }
            else if (secion != null)
            {
                key += secion;
            }
            else
            {
                throw new Exception("Could not define section name.");
            }

            if (subSection != null && this.SubSection != null)
            {
                if (subSection != this.SubSection)
                {
                    throw new Exception("SubSection name should be same.");
                }
            }
            else if (this.SubSection != null)
            {
                key += "." + this.SubSection;
            }
            else if (subSection != null)
            {
                key += "." + subSection;
            }

            return key;
        }

        /// <summary>
        /// エントリからセクション名を取得します。セクション名を収めるプロパティが存在しない場合、null を返します。
        /// </summary>
        /// <param name="entry"></param>
        /// <returns></returns>
        private string GetSection(TEntry entry)
        {
            var sectionInfo = GetSectionAttributeInfo(entry);

            if (sectionInfo != null)
            {
                return (string)typeof(TEntry).InvokeMember(sectionInfo.PropertyInfo.Name, System.Reflection.BindingFlags.GetProperty, null, entry, null);
            }

            return null;
        }

        /// <summary>
        /// エントリにセクション名を設定します。セクション名を収めるプロパティが存在しない場合、何も行いません。
        /// </summary>
        /// <param name="entry"></param>
        /// <param name="section"></param>
        private void SetSection(TEntry entry, string section)
        {
            var sectionInfo = GetSectionAttributeInfo(entry);

            if (sectionInfo != null)
            {
                typeof(TEntry).InvokeMember(sectionInfo.PropertyInfo.Name, System.Reflection.BindingFlags.SetProperty, null, entry, new[] { section });
            }
        }

        /// <summary>
        /// エントリからサブセクション名を取得します。サブセクション名を収めるプロパティが存在しない場合、null を返します。
        /// </summary>
        /// <param name="entry"></param>
        /// <returns></returns>
        private string GetSubSection(TEntry entry)
        {
            var subSectionInfo = GetSubSectionAttributeInfo(entry);

            if (subSectionInfo != null)
            {
                return (string)typeof(TEntry).InvokeMember(subSectionInfo.PropertyInfo.Name, System.Reflection.BindingFlags.GetProperty, null, entry, null);
            }

            return null;
        }

        /// <summary>
        /// エントリにサブセクション名を設定します。セクション名を収めるプロパティが存在しない場合、何も行いません。
        /// </summary>
        /// <param name="entry"></param>
        /// <param name="subSection"></param>
        private void SetSubSection(TEntry entry, string subSection)
        {
            var subSectionInfo = GetSubSectionAttributeInfo(entry);

            if (subSectionInfo != null)
            {
                typeof(TEntry).InvokeMember(subSectionInfo.PropertyInfo.Name, System.Reflection.BindingFlags.SetProperty, null, entry, new[] { subSection });
            }
        }

        private EntryPropertyAttributeInfo<EntrySectionAttribute> GetSectionAttributeInfo(TEntry entry)
        {
            var sections = typeof(TEntry).GetProperties().Select(x => new EntryPropertyAttributeInfo<EntrySectionAttribute>
            {
                PropertyInfo = x,
                Attribute = (EntrySectionAttribute)Attribute.GetCustomAttribute(x, typeof(EntrySectionAttribute))
            }).Where(x => x.Attribute != null);

            if(sections.Count() > 0)
            {
                return sections.ElementAt(0);
            }
            else
            {
                return null;
            }
        }

        private EntryPropertyAttributeInfo<EntrySubSectionAttribute> GetSubSectionAttributeInfo(TEntry entry)
        {
            var sections = typeof(TEntry).GetProperties().Select(x => new EntryPropertyAttributeInfo<EntrySubSectionAttribute>
            {
                PropertyInfo = x,
                Attribute = (EntrySubSectionAttribute)Attribute.GetCustomAttribute(x, typeof(EntrySubSectionAttribute))
            }).Where(x => x.Attribute != null);

            if (sections.Count() > 0)
            {
                return sections.ElementAt(0);
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// 指定の型に定義された属性を得ます。
        /// </summary>
        /// <returns>エントリの属性</returns>
        private IEnumerable<EntryPropertyAttributeInfo<EntryMemberAttribute>> GetEntryPropertyAttributeInfos()
        {
            return typeof(TEntry).GetProperties().Select(x => new EntryPropertyAttributeInfo<EntryMemberAttribute>
            {
                PropertyInfo = x,
                Attribute = (EntryMemberAttribute)Attribute.GetCustomAttribute(x, typeof(EntryMemberAttribute))
            }).Where(x => x.Attribute != null);
        }

        /// <summary>
        /// 指定の型を書き出し用のキーバリュー形式に変換する
        /// </summary>
        /// <param name="entry"></param>
        /// <returns></returns>
        private GitConfigSection ConvertToGitConfigSection(TEntry entry)
        {
            var configSection = new GitConfigSection();

            // セクション、サブセクションの読み出し
            configSection.Section = GetSection(entry);
            configSection.SubSection = GetSubSection(entry);

            foreach (var info in GetEntryPropertyAttributeInfos())
            {
                var value = typeof(TEntry).InvokeMember(info.PropertyInfo.Name, System.Reflection.BindingFlags.GetProperty, null, entry, null);

                if (!info.Attribute.IsRequired && object.Equals(value, info.Attribute.DefaultValue))
                {
                    // メンバが必須ではなく、デフォルト値と等しい場合、書き出しは行わない。
                    continue;
                }

                if (value.GetType().IsEnum ||
                    value.GetType() == typeof(bool))
                {
                    // Enum は、小文字のみの文字列で書き出す。
                    // Boolean は、小文字のみの文字列で書き出す。
                    value = value.ToString().ToLower();
                }
                else if (value.GetType() == typeof(string))
                {
                    // 文字列は文字列のまま扱う
                }
                else
                {
                    // その他の型はサポート外
                    throw new NotImplementedException(string.Format("typeof({0}) does not supported.", value.GetType()));
                }

                configSection.Entries.Add(info.Attribute.EntryName, (string)value);
            }

            return configSection;
        }

        /// <summary>
        /// キーバリュー形式のデータから指定の型に変換する
        /// </summary>
        /// <param name="values"></param>
        /// <returns></returns>
        private TEntry ConvertToEntry(GitConfigSection configSection)
        {
            var entry = new TEntry();

            // セクション、サブセクションの設定
            SetSection(entry, configSection.Section);
            SetSubSection(entry, configSection.SubSection);

            foreach (var info in GetEntryPropertyAttributeInfos())
            {
                if (configSection.Entries.ContainsKey(info.Attribute.EntryName))
                {
                    var str = configSection.Entries[info.Attribute.EntryName];
                    object value = str;

                    if (info.PropertyInfo.PropertyType == typeof(string))
                    {
                        // そのまま渡す
                        value = str;
                    }
                    else if (info.PropertyInfo.PropertyType.IsEnum)
                    {
                        // Enum は、大文字の大小を無視して変換を行う
                        value = Enum.Parse(info.PropertyInfo.PropertyType, str, true);
                    }
                    else if (info.PropertyInfo.PropertyType.IsPrimitive)
                    {
                        // プリミティブ型は、静的な Parse メソッドによって変換を行う
                        var parser = info.PropertyInfo.PropertyType.GetMethod("Parse");
                        value = parser.Invoke(null, new[] { str });
                    }

                    typeof(TEntry).InvokeMember(info.PropertyInfo.Name, System.Reflection.BindingFlags.SetProperty, null, entry, new[] { value });
                }
                else if (!info.Attribute.IsRequired)
                {
                    // エントリが必須ではなく、キーバリュー上に見つからない場合は、デフォルト値を使用する
                    typeof(TEntry).InvokeMember(info.PropertyInfo.Name, System.Reflection.BindingFlags.SetProperty, null, entry, new[] { info.Attribute.DefaultValue });
                }
                else
                {
                    // 必須のエントリが見つからなかった
                    throw new ConfigRequiredMemberNotFoundException(string.Format("Required member \"{0}\" is not found\n", info.Attribute.EntryName));
                }
            }

            return entry;
        }
    }
}
