﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using GitExternalRepository.Configs;
using GitExternalRepository.Consoles;
using GitExternalRepository.Extensions;
using GitExternalRepository.Repository;
using Nintendo.Foundation.IO;
using System.Collections.Concurrent;
using GitExternalRepository.Exstorage;

namespace GitExternalRepository.Commands
{
    /// <summary>
    /// 外部リポジトリの更新を行うコマンドです。
    /// </summary>
    [CommandDefinition(typeof(Parameters))]
    public class UpdateCommand : CommandBase
    {
        public class Parameters
        {
            [CommandLineOption('f', "force", Description = "Throw away local changes and update the repository to the revision.", IsRequired = false)]
            public bool Force { get; set; }

            [CommandLineValue(0, ValueName = "path", Description = "Specify an external repository directory.", IsRequired = false)]
            public string Path { get; set; }

            [CommandLineOption('j', "jobs", Description = "Specifies the number of jobs to run simultaneously.", IsRequired = false)]
            public int Jobs { get; set; }
        }

        public UpdateCommand()
        {
        }

        public UpdateCommand(ConsoleBase console)
            : base(console)
        {
        }

        public override bool Execute(object parameters)
        {
            var commandParameters = (Parameters)parameters;

            var activeEntries = Environments.ActiveRepositoryList.Read();

            var repositoryList = Environments.RepositoryList;
            var entries = repositoryList.Exists() ? repositoryList.Read() : new List<RepositoryListEntry>();

            if (commandParameters.Path == null)
            {
                var exceptions = new ConcurrentQueue<Exception>();

                var options = new ParallelOptions();
                options.MaxDegreeOfParallelism = (commandParameters.Jobs > 0) ? commandParameters.Jobs : 1;
                // アクティブリポジトリリスト上の全エントリを更新します。
                Parallel.ForEach(activeEntries, options, e =>
                {
                    try
                    {
                        var entry = entries.FirstOrDefault(x => x.TargetDirectory == e.TargetDirectory);
                        if (entry != null)
                        {
                            this.UpdateEntry(entry, commandParameters.Force);
                        }
                        else
                        {
                            // リビジョンの切り替えにより、アクティブリポジトリリストには存在するが、リポジトリリストには存在しない状態が起こりえる。
                            // このため、リポジトリリストにエントリが見つからない場合は、これを警告するメッセージを表示しつつ、処理を続行する。
                            this.CommandConsole.WriteLineToOut(string.Format("Info: '{0}' does not currently exist in the repository list '{1}'.", e.TargetDirectory, repositoryList.FilePath));
                        }
                    }
                    catch (Exception ex)
                    {
                        if (options.MaxDegreeOfParallelism > 1)
                        {
                            exceptions.Enqueue(ex);
                        }
                        else
                        {
                            throw;
                        }
                    }
                });

                if (exceptions.Count > 0)
                {
                    throw new AggregateException(exceptions);
                }
            }
            else
            {
                // 指定のディレクトリだけ更新します。

                // 与えられた引数
                var path = commandParameters.Path.TrimSlash();
                // 親リポジトリのルートからの相対パス
                var relatedPathFromRoot = PathUtility.GetRelativePath(Environments.GitRepository.GetRepositoryRoot() + @"\", path);

                var entry = entries.FirstOrDefault(x => x.TargetDirectory == relatedPathFromRoot);
                if (entry != null)
                {
                    if (activeEntries.Any(x => x.TargetDirectory == relatedPathFromRoot))
                    {
                        this.UpdateEntry(entry, commandParameters.Force);
                    }
                    else
                    {
                        throw new GitExternalRepositoryException(string.Format("'{0}' is not initialized.", relatedPathFromRoot));
                    }
                }
                else
                {
                    throw new GitExternalRepositoryException(string.Format("'{0}' does not exist in the repository list '{1}'.", relatedPathFromRoot, repositoryList.FilePath));
                }
            }

            return true;
        }

        /// <summary>
        /// 指定のエントリの外部リポジトリをエントリに記述されたリビジョンで更新します。
        /// </summary>
        /// <param name="entry">リポジトリリストエントリ</param>
        private void UpdateEntry(RepositoryListEntry entry, bool force)
        {
            // リポジトリ URL
            var url = entry.Url;
            // ディレクトリパス
            var path = PathUtility.Combine(Environments.GitRepository.GetRepositoryRoot(), entry.TargetDirectory);
            // リビジョン
            var revision = entry.Revision;
            // 親リポジトリのルートからの相対パス = エントリに記述されたパス
            var relatedPathFromRoot = entry.TargetDirectory;
            // exstorage を利用するリポジトリかどうか
            var useExstorage = entry.UseExstorage;

            // 外部リポジトリの GitRepository インスタンス
            var exrepo = Environments.RepositoryFactory.Create(entry.RepositoryType, path);

            this.CommandConsole.WriteLineToOut("Update '{0}'.", relatedPathFromRoot);

            // ディレクトリが存在しない、または、そのディレクトリがリポジトリルートでない場合は、
            // exrepo でチェックアウトしたリポジトリが存在していないとみなし、クローンを試みます。
            if (!Directory.Exists(path) || !exrepo.IsRepositoryRoot())
            {
                // .git を配置する場所を取得します。
                var gitDirectoryPath = PathUtility.Combine(Environments.GetModuleDirectoryPath(), relatedPathFromRoot);

                this.CommandConsole.WriteLineToOut("Cloning '{0}' of '{1}'.", revision, url);

                if (force)
                {
                    if (exrepo.CanSeparateMetaDirectory())
                    {
                        var meta = Environments.RepositoryFactory.Create(entry.RepositoryType, gitDirectoryPath);

                        if (Directory.Exists(gitDirectoryPath) && !meta.IsMetaDirectory())
                        {
                            // メタディレクトリではないものが置かれているので、これを削除
                            Directory.Delete(gitDirectoryPath, true);
                        }
                    }

                    // ワークツリー展開先に置かれているものを削除
                    if(Directory.Exists(path))
                    {
                        Directory.Delete(path, true);
                    }
                    else if(File.Exists(path))
                    {
                        File.Delete(path);
                    }
                }

                try
                {
                    if (exrepo.CanSeparateMetaDirectory())
                    {
                        exrepo.Clone(url, revision, gitDirectoryPath);
                    }
                    else
                    {
                        exrepo.Clone(url, revision);
                    }
                }
                catch (RepositoryOperationFailedException e)
                {
                    throw new GitExternalRepositoryException(
                        string.Format("Failed to clone '{0}' of '{1}' into '{2}'", revision, url, relatedPathFromRoot), e);
                }
            }
            else
            {
                // 指定のリビジョンをチェックアウトします。
                this.CommandConsole.WriteLineToOut("Checkouting the revision '{0}' of '{1}'.", revision, url);

                try
                {
                    exrepo.Checkout(revision, force);
                }
                catch (RepositoryOperationFailedException e)
                {
                    throw new GitExternalRepositoryException(
                        string.Format("Failed to clone the revision '{0}' of {1} into '{2}'", revision, url, relatedPathFromRoot), e);
                }
            }

            // exstorage を利用するための設定を行います。
            if (useExstorage)
            {
                var manager = new ExstorageManager();

                try
                {
                    this.CommandConsole.WriteLineToOut("Install exstorage alias from the parent repository.");
                    var aliasInstalled = manager.InstallAliasFromParent(exrepo, Environments.GitRepository);

                    this.CommandConsole.WriteLineToOut("Copy exstorage settings from the parent repository.");
                    var settingsCopied = manager.CopyExstorageSettingsFromParent(relatedPathFromRoot, exrepo, Environments.GitRepository);

                    if (aliasInstalled)
                    {
                        // コマンドが存在する状態のときだけ、フィルタのインストールを試みます。
                        this.CommandConsole.WriteLineToOut("Install exstorage filter if the parent repository has.");

                        manager.InstallFilterIfParentHas(exrepo, Environments.GitRepository);

                        // コマンドが存在する状態のときだけ、フックスクリプトのインストールを試みます。
                        this.CommandConsole.WriteLineToOut("Install exstorage hook script.");

                        manager.InstallHookScript(exrepo);
                    }

                    if (aliasInstalled && settingsCopied)
                    {
                        // コマンドが存在する状態であり、exstorage 設定が新規あるいは変更点によりコピーされた場合のみ、
                        // 展開されていないハッシュ値が存在している可能性があるため、 exstorage expand 操作を実行します。
                        this.CommandConsole.WriteLineToOut("Expanding exstorage files.");

                        var outputs = manager.Expand(exrepo);

                        this.CommandConsole.WriteLineToOut("{0}\n", outputs);
                    }
                }
                catch (ExstorageManagerOperationFailedException e)
                {
                    throw new ExstorageManagerOperationFailedException(string.Format("Failed to setup to use exstorage in '{0}'", relatedPathFromRoot), e);
                }
            }
        }
    }
}
