﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Text.RegularExpressions;

using Nintendo.ToolFoundation.CommandLine;

namespace Bin2Cpp
{
    internal class CommandLineParams
    {
        [CommandLineValueOption(
            'i',
            "input",
            IsRequired = true,
            Description = "Input binary file path",
            ValueName = "INPUT_PATH")]
        public string InputPath { get; set; } = null;
        [CommandLineValueOption(
            "output-header",
            IsRequired = true,
            Description = "Output header file path",
            ValueName = "OUTPUT_HEADER_PATH")]
        public string OutputHeaderPath { get; set; }
        [CommandLineValueOption(
            "output-cpp",
            IsRequired = true,
            Description = "Output cpp file path.",
            ValueName = "OUTPUT_CPP_PATH-cpp-path")]
        public string OutputCppPath { get; set; }
        [CommandLineValueOption(
            'b',
            "begin",
            IsRequired = true,
            Description = "Set symbol name for top of binary.",
            ValueName = "SYMBOL_BEGIN")]
        public string SymbolNameBegin { get; set; }
        [CommandLineValueOption(
            'e',
            "end",
            IsRequired = true,
            Description = "Set symbol name for top of binary.",
            ValueName = "SYMBOL_END")]
        public string SymbolNameEnd { get; set; }
        [CommandLineValueOption(
            'n',
            "name-space",
            IsRequired = false,
            Description = "Set namespace for symbol.",
            ValueName = "NAMESPACE")]
        public string NameSpace { get; set; }
        [CommandLineValueOption(
            'a',
            "align",
            IsRequired = false,
            Description = "Set alignment.",
            ValueName = "ALIGN")]
        public int Align { get; set; }
        [CommandLineFlagOption(
            'c',
            "const",
            Description = "Use const qualifier.")]
        public bool Const { get; set; }
    }

    internal static class CommandLineParamsParser
    {
        /// <summary>
        /// 与えられた文字列が C++ の識別子として有効か判定します。
        /// universal-character-name には非対応です。
        /// </summary>
        /// <param name="name">判定対象の文字列</param>
        /// <returns>C++ の識別子として有効なら true</returns>
        private static bool IsValidCppIdentifier(string name)
        {
            if (name == null || name.Length == 0)
            {
                return false;
            }

            var re = new Regex("[A-Za-z_][A-Za-z0-9_]*");
            return re.Match(name).Success;
        }

        /// <summary>
        /// 与えられた文字列が C++ の名前空間名として有効か判定します。
        /// universal-character-name には非対応です。
        /// </summary>
        /// <param name="name">判定対象の文字列</param>
        /// <returns>C++ の名前空間名として有効なら true</returns>
        private static bool IsValidNameSpace(string name)
        {
            var tokens = new List<string>();

            // ':' で構成されるか否かで tokenize
            for (int head = 0, tail = 0; head < name.Length; head = tail)
            {
                if (name[head] == ':')
                {
                    for (tail = head; tail < name.Length; tail++)
                    {
                        if (name[tail] != ':')
                        {
                            break;
                        }
                    }
                }
                else
                {
                    for (tail = head; tail < name.Length; tail++)
                    {
                        if (name[tail] == ':')
                        {
                            break;
                        }
                    }
                }
                tokens.Add(name.Substring(head, tail - head));
            }

            for (int i = 0; i < tokens.Count; i++)
            {
                if (i == tokens.Count - 1)
                {
                    // 末尾の token は c++ の識別子 ("::" は不可)
                    if (!IsValidCppIdentifier(tokens[i]))
                    {
                        return false;
                    }
                }
                else
                {
                    // それ以外の token は c++ の識別子か "::"
                    if (!(IsValidCppIdentifier(tokens[i]) || tokens[i] == "::"))
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        internal static CommandLineParams Parse(string[] args)
        {
            CommandLineParams appParams = null;
            var result = CommandLine.ParseArgs(args, out appParams);

            switch (result)
            {
                case ParseResultType.Failure:
                    Console.WriteLine("--");
                    Console.WriteLine(CommandLine.GetHelpText<CommandLineParams>());
                    return null;

                case ParseResultType.Help:
                case ParseResultType.Version:
                    return null;
            }

            // Validate
            bool isInvalidArg = false;

            if (!File.Exists(appParams.InputPath))
            {
                Console.WriteLine($"File {appParams.InputPath} is not exist.");
                isInvalidArg = true;
            }
            if (!IsValidCppIdentifier(appParams.SymbolNameBegin))
            {
                Console.WriteLine($"Name {appParams.SymbolNameBegin} is invalid.");
                isInvalidArg = true;
            }
            if (!IsValidCppIdentifier(appParams.SymbolNameEnd))
            {
                Console.WriteLine($"Name {appParams.SymbolNameEnd} is invalid.");
                isInvalidArg = true;
            }
            if (appParams.NameSpace != null && !IsValidNameSpace(appParams.NameSpace))
            {
                Console.WriteLine($"Namespace {appParams.NameSpace} is invalid.");
                isInvalidArg = true;
            }

            if (isInvalidArg)
            {
                return null;
            }
            return appParams;
        }
    }
}
