﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/utilTool/utilTool_CommandInterface.h>
#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_StringUtility.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>
#include <algorithm>
#include <cstdarg>
#include <nn/nn_Abort.h>
#include <string>
#include <cstdlib>

namespace nn {
    namespace utilTool {


        void ArgumentParseStatus::SetCondition(bool isRequired, bool isAllowedMultiple)
        {
            m_IsRequired = isRequired;
            m_IsAllowedMultiple = isAllowedMultiple;
        }

        void ArgumentParseStatus::ResetStatus()
        {
            m_FoundCount = 0;
        }

        void ArgumentParseStatus::NotifyFoundArgument(std::string argument)
        {
            m_FoundCount += 1;
            m_LastArgument = argument;
        }

        bool ArgumentParseStatus::ValidateStatus()
        {
            if (m_IsRequired && m_FoundCount == 0)
            {
                NN_UTILTOOL_LOG_ERROR("Argument(%s) is not found.", m_Name.c_str());
                return false;
            }

            if (!m_IsAllowedMultiple && 1 < m_FoundCount)
            {
                NN_UTILTOOL_LOG_ERROR("Argument(%s) is duplicated. value = '%s'", m_Name.c_str(), m_LastArgument.c_str());
                return false;
            }

            return true;
        }

        bool ArgumentParseStatus::IsFound()
        {
            return 0 < m_FoundCount;
        }

        void StringArgumentStore::Reset()
        {
            *m_pOut = m_DefaultValue;
        }

        bool StringArgumentStore::Store(std::string value)
        {
            *m_pOut = value;

            return true;
        }

        void IntegerArgumentStore::Reset()
        {
            *m_pOut = m_DefaultValue;
        }

        bool IntegerArgumentStore::Store(std::string value)
        {
            *m_pOut = std::atoi(value.c_str());

            return true;
        }

        void Integer64ArgumentStore::Reset()
        {
            *m_pOut = m_DefaultValue;
        }

        bool Integer64ArgumentStore::Store(std::string value)
        {
            // strtoll 系の関数がうまく使えないので一時的に適当な実装の変換関数にしておく
            // ::strtoll は cstdlib を使えとコーディングチェックで弾かれる
            // std::strtoll は std::stoll を使えとコンパイラに弾かれる
            // std::stoll は 定義が無いとコンパイラに弾かれる
            // *m_pOut = std::stoll(value.c_str(), nullptr, 10);

            const char *end;
            *m_pOut = StrToInt64(value.c_str(), &end, 10);

            return *end == '\0';
        }

        void MultiStringArgumentStore::Reset()
        {
            m_pOut->clear();
        }

        bool MultiStringArgumentStore::Store(std::string value)
        {
            m_pOut->push_back(value);
            return true;
        }

        void MultiIntegerArgumentStore::Reset()
        {
            m_pOut->clear();
        }

        bool MultiIntegerArgumentStore::Store(std::string value)
        {
            m_pOut->push_back(std::atoi(value.c_str()));
            return true;
        }

        void MultiInteger64ArgumentStore::Reset()
        {
            m_pOut->clear();
        }

        bool MultiInteger64ArgumentStore::Store(std::string value)
        {
            // strtoll 系の関数がうまく使えないので一時的に適当な実装の変換関数にしておく
            // ::strtoll は cstdlib を使えとコーディングチェックで弾かれる
            // std::strtoll は std::stoll を使えとコンパイラに弾かれる
            // std::stoll は 定義が無いとコンパイラに弾かれる
            // *m_pOut = std::stoll(value.c_str(), nullptr, 10);

            const char *end;
            m_pOut->push_back(StrToInt64(value.c_str(), &end, 10));

            return *end == '\0';
        }

        void BoolFlagArgumentStore::Reset()
        {
            *m_pOut = false;
        }

        bool BoolFlagArgumentStore::Store()
        {
            *m_pOut = true;
            return true;
        }

        bool NullArgumentStore::Store(std::string value)
        {
            NN_UTILTOOL_LOG_ERROR("Unexpected argument: %s\n", value.c_str());
            return false;
        }

        bool KeywordArgumentName::IsMatch(std::string argument)
        {
            if (m_ShortName != '\0' && IsMatchImpl(argument, std::string("-") + std::string(&m_ShortName, 1)))
            {
                return true;
            }

            if (m_LongName != nullptr && IsMatchImpl(argument, std::string("--") + std::string(m_LongName)))
            {
                return true;
            }

            return false;
        }

        bool KeywordArgumentName::IsMatchImpl(const std::string& argument, const std::string& targetName)
        {
            if (argument.length() >= targetName.length() && argument.compare(0, targetName.length(), targetName.c_str()) == 0)
            {
                if (argument.length() == targetName.length())
                {
                    return true;
                }

                // '=' 区切りで値を持つ場合
                if (argument[targetName.length()] == '=')
                {
                    return true;
                }
            }

            return false;
        }

        bool KeywordArgument::GetArgumentValue(std::string argument, std::string* pOutValue)
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);

            // 区切り文字 '=' より後ろを抽出する
            // 引数の値にダブルクォートを含めた場合、ダブルクォートは main() の手前でトリミングされているはずなので、
            // ここでは何もしない
            auto separatorPosition = argument.find_first_of('=');

            if (separatorPosition == std::string::npos)
            {
                return false;
            }

            *pOutValue = argument.substr(separatorPosition + 1);

            return true;
        }

        void SingleCommandInterface::AddHiPriorityFlagArgument(FlagArgument argument, KeywordArgumentDocument document)
        {
            m_CommandDocument.AddKeywordArgument(document);
            argument.SetCondition(false, false);
            argument.SetName(document.GetDisplayName());
            m_HiPriorityFlagArguments.push_back(argument);
        }

        void SingleCommandInterface::AddFlagArgument(FlagArgument argument, KeywordArgumentDocument document)
        {
            m_CommandDocument.AddKeywordArgument(document);
            argument.SetCondition(false, false);
            argument.SetName(document.GetDisplayName());
            m_FlagArguments.push_back(argument);
        }

        void SingleCommandInterface::AddKeywordArgument(KeywordArgument argument, KeywordArgumentDocument document)
        {
            m_CommandDocument.AddKeywordArgument(document);
            argument.SetCondition(document.IsRequired(), false);
            argument.SetName(document.GetDisplayName());
            m_KeywordArguments.push_back(argument);
        }

        void SingleCommandInterface::AddPositionalArgument(PositionalArgument argument, PositionalArgumentDocument document)
        {
            m_CommandDocument.AddPositionalArgument(document);
            argument.SetCondition(document.IsRequired(), false);
            argument.SetName(document.GetPlaceHolderName());
            m_PositionalArguments.push_back(argument);
        }

        void SingleCommandInterface::SetVariableArgument(VariableArguments argument, VariableArgumentDocument document)
        {
            m_CommandDocument.SetVariableArgument(document);
            argument.SetCondition(false, false);
            argument.SetName(document.GetPlaceHolderName());
            m_VariableArguments = argument;
        }

        template<> void SingleCommandInterface::AddFlagArgument<bool>(bool *pOut, char shortName, const char* longName, const char* description)
        {
            AddFlagArgument(
                FlagArgument(std::shared_ptr<FlagArgumentStore>(new BoolFlagArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordFlagArgument(shortName, longName, "arg", description, false));
        }

        template<> void SingleCommandInterface::AddKeywordArgument<std::string>(std::string *pOut, char shortName, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new StringArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, isRequired));
        }

        template<> void SingleCommandInterface::AddKeywordArgument<int>(int *pOut, char shortName, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new IntegerArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, isRequired));
        }

        template<> void SingleCommandInterface::AddKeywordArgument<int64_t>(int64_t *pOut, char shortName, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new Integer64ArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, isRequired));
        }

        template<> void SingleCommandInterface::AddKeywordArgument<std::vector<std::string>>(std::vector<std::string> *pOut, char shortName, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new MultiStringArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, isRequired));
        }

        template<> void SingleCommandInterface::AddKeywordArgument<std::vector<int>>(std::vector<int> *pOut, char shortName, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new MultiIntegerArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, isRequired));
        }

        template<> void SingleCommandInterface::AddKeywordArgument<std::vector<int64_t>>(std::vector<int64_t> *pOut, char shortName, const char* longName, const char* description, bool isRequired)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new MultiInteger64ArgumentStore(pOut)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, isRequired));
        }

        template<> void SingleCommandInterface::AddKeywordArgumentWithDefaultValue<std::string>(std::string *pOut, char shortName, const char* longName, const char* description, std::string defaultValue)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new StringArgumentStore(pOut, defaultValue)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, false));
        }

        template<> void SingleCommandInterface::AddKeywordArgumentWithDefaultValue<int>(int *pOut, char shortName, const char* longName, const char* description, int defaultValue)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new IntegerArgumentStore(pOut, defaultValue)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, false));
        }

        template<> void SingleCommandInterface::AddKeywordArgumentWithDefaultValue<int64_t>(int64_t *pOut, char shortName, const char* longName, const char* description, int64_t defaultValue)
        {
            AddKeywordArgument(
                KeywordArgument(std::shared_ptr<ValueArgumentStore>(new Integer64ArgumentStore(pOut, defaultValue)), KeywordArgumentName(shortName, longName)),
                KeywordArgumentDocument::MakeKeywordValueArgument(shortName, longName, "arg", description, false));
        }

        template<> void SingleCommandInterface::AddPositionalArgument<std::string>(std::string *pOut, int index, const char* name, const char* description, bool isRequired)
        {
            AddPositionalArgument(
                PositionalArgument(std::shared_ptr<ValueArgumentStore>(new StringArgumentStore(pOut)), index, name),
                PositionalArgumentDocument(index, name, description, isRequired));
        }

        template<> void SingleCommandInterface::AddPositionalArgument<int>(int *pOut, int index, const char* name, const char* description, bool isRequired)
        {
            AddPositionalArgument(
                PositionalArgument(std::shared_ptr<ValueArgumentStore>(new IntegerArgumentStore(pOut)), index, name),
                PositionalArgumentDocument(index, name, description, isRequired));
        }

        template<> void SingleCommandInterface::AddPositionalArgument<int64_t>(int64_t *pOut, int index, const char* name, const char* description, bool isRequired)
        {
            AddPositionalArgument(
                PositionalArgument(std::shared_ptr<ValueArgumentStore>(new Integer64ArgumentStore(pOut)), index, name),
                PositionalArgumentDocument(index, name, description, isRequired));
        }

        template<> void SingleCommandInterface::SetVariableArguments<std::string>(std::vector<std::string> *pOut, const char* name, const char* description, bool isRequired)
        {
            SetVariableArgument(
                VariableArguments(std::shared_ptr<ValueArgumentStore>(new MultiStringArgumentStore(pOut)), name),
                VariableArgumentDocument(name, description, isRequired));
        }

        bool SingleCommandInterface::TryParseVa(int argumentCount, ...)
        {
            std::va_list argumentList;
            va_start(argumentList, argumentCount);
            NN_UTIL_SCOPE_EXIT{ va_end(argumentList); };

            return TryParseVaList(argumentCount, argumentList);
        }

        bool SingleCommandInterface::TryParseVaList(int argumentCount, std::va_list arguments)
        {
            std::vector<std::string> argumentList;

            argumentList.resize(argumentCount);

            for (int i = 0; i < argumentCount; i++)
            {
                argumentList[i] = std::string(va_arg(arguments, char*));
            }

            return TryParse(argumentList);
        }

        bool SingleCommandInterface::TryParse(int argumentCount, const char** argumentValues)
        {
            std::vector<std::string> argumentList;

            argumentList.resize(argumentCount);

            for (int i = 0; i < argumentCount; i++)
            {
                argumentList[i] = std::string(argumentValues[i]);
            }

            return TryParse(argumentList);
        }

        bool SingleCommandInterface::TryParse(std::vector<std::string> arguments)
        {
            Reset();

            std::vector<std::string> argumentsExceptHiPriority;
            if (!TryParseFlagArguments(&argumentsExceptHiPriority, arguments, m_HiPriorityFlagArguments))
            {
                return false;
            }

            for (auto argDef : m_HiPriorityFlagArguments)
            {
                if (argDef.IsFound())
                {
                    return true;
                }
            }

            std::vector<std::string> argumentsExceptFlags;
            if (!TryParseFlagArguments(&argumentsExceptFlags, argumentsExceptHiPriority, m_FlagArguments))
            {
                return false;
            }

            std::vector<std::string> positionalArguments;
            if (!TryParseKeywordArguments(&positionalArguments, argumentsExceptFlags, m_KeywordArguments))
            {
                return false;
            }

            std::vector<std::string> restArguments;
            if (!TryParsePositionalArguments(&restArguments, positionalArguments, m_PositionalArguments))
            {
                return false;
            }

            if (!TryParseVariableArguments(restArguments))
            {
                return false;
            }

            if (!ValidateAfterParse())
            {
                return false;
            }

            return true;
        }

        bool SingleCommandInterface::TryParseFlagArguments(std::vector<std::string> *pOutRestArguments, std::vector<std::string> &arguments, std::vector<FlagArgument> &argumentDefinitions)
        {
            for (auto argument : arguments)
            {
                std::vector<FlagArgument>::iterator pFlagArgument;
                if (FindFlagArgument(&pFlagArgument, argument, argumentDefinitions))
                {
                    pFlagArgument->NotifyFoundArgument(argument);

                    if (!pFlagArgument->ValidateStatus())
                    {
                        return false;
                    }

                    if (!pFlagArgument->GetStore()->Store())
                    {
                        NN_UTILTOOL_LOG_ERROR("Failed parse: KeywordArgumentName = '%s'", argument.c_str());
                        return false;
                    }
                }
                else
                {
                    pOutRestArguments->push_back(argument);
                }
            }

            return true;
        }

        bool SingleCommandInterface::TryParseKeywordArguments(std::vector<std::string> *pOutRestArguments, std::vector<std::string> &arguments, std::vector<KeywordArgument> &argumentDefinitions)
        {
            enum ParseState
            {
                ParseState_Neutral,
                ParseState_ExpectingValue
            };

            ParseState state = ParseState_Neutral;
            std::string KeywordArgumentName;
            std::shared_ptr<ValueArgumentStore> argumentExpectValue;

            for (auto argument : arguments)
            {
                if (state == ParseState_Neutral)
                {
                    std::vector<KeywordArgument>::iterator pKeywordArgument;
                    if (FindKeywordArgument(&pKeywordArgument, argument, argumentDefinitions))
                    {
                        pKeywordArgument->NotifyFoundArgument(argument);

                        if (!pKeywordArgument->ValidateStatus())
                        {
                            return false;
                        }

                        // 引数の値が ' ' or '=' のどちらで区切られているかで分岐
                        std::string argumentValue;
                        if (pKeywordArgument->GetArgumentValue(argument, &argumentValue))
                        {
                            if (!pKeywordArgument->GetStore()->Store(argumentValue))
                            {
                                NN_UTILTOOL_LOG_ERROR("Failed parse : KeywordArgument = '%s'", argument.c_str());
                                return false;
                            }
                        }
                        else
                        {
                            argumentExpectValue = pKeywordArgument->GetStore();
                            KeywordArgumentName = argument;
                            state = ParseState_ExpectingValue;
                        }
                    }
                    else
                    {
                        pOutRestArguments->push_back(argument);
                    }
                }
                else if (state == ParseState_ExpectingValue)
                {
                    if (!argumentExpectValue->Store(argument))
                    {
                        NN_UTILTOOL_LOG_ERROR("Failed parse : KeywordArgumentName = '%s', value = '%s'", KeywordArgumentName.c_str(), argument.c_str());
                        return false;
                    }

                    state = ParseState_Neutral;
                }
                else
                {
                    NN_ABORT("Unknown status: %d", state);
                }
            }

            if (state != ParseState_Neutral)
            {
                NN_UTILTOOL_LOG_ERROR("Failed parse: value not found. KeywordArgumentName = '%s'", KeywordArgumentName.c_str());
                return false;
            }

            return true;
        }

        bool SingleCommandInterface::TryParsePositionalArguments(std::vector<std::string> *pOutRestArguments, std::vector<std::string> &arguments, std::vector<PositionalArgument> &argumentDefinitions)
        {
            int index = 0;
            for (auto argument : arguments)
            {
                std::vector<PositionalArgument>::iterator pPositionalArgument;
                if (FindPositionalArgument(&pPositionalArgument, index, argumentDefinitions))
                {
                    pPositionalArgument->NotifyFoundArgument(argument);

                    if (!pPositionalArgument->ValidateStatus())
                    {
                        return false;
                    }

                    if (!pPositionalArgument->GetStore()->Store(argument))
                    {
                        return false;
                    }
                }
                else
                {
                    pOutRestArguments->push_back(argument);
                }

                index += 1;
            }
            return true;
        }

        bool SingleCommandInterface::TryParseVariableArguments(std::vector<std::string> &arguments)
        {
            for (auto argument : arguments)
            {
                if (!m_VariableArguments.GetStore()->Store(argument))
                {
                    return false;
                }
            }
            return true;
        }

        void SingleCommandInterface::Reset()
        {
            for (auto flagArgument = m_FlagArguments.begin(); flagArgument != m_FlagArguments.end(); flagArgument++)
            {
                flagArgument->ResetStatus();
                flagArgument->GetStore()->Reset();
            }

            for (auto keywordArgument = m_KeywordArguments.begin(); keywordArgument != m_KeywordArguments.end(); keywordArgument++)
            {
                keywordArgument->ResetStatus();
                keywordArgument->GetStore()->Reset();
            }

            for (auto positionalArgument = m_PositionalArguments.begin(); positionalArgument != m_PositionalArguments.end(); positionalArgument++)
            {
                positionalArgument->ResetStatus();
                positionalArgument->GetStore()->Reset();
            }

            m_VariableArguments.ResetStatus();
        }

        bool SingleCommandInterface::ValidateAfterParse()
        {
            for (auto flagArgument : m_FlagArguments)
            {
                if (!flagArgument.ValidateStatus())
                {
                    return false;
                }
            }

            for (auto keywordArgument : m_KeywordArguments)
            {
                if (!keywordArgument.ValidateStatus())
                {
                    return false;
                }
            }

            for (auto positionalArgument : m_PositionalArguments)
            {
                if (!positionalArgument.ValidateStatus())
                {
                    return false;
                }
            }

            if (!m_VariableArguments.ValidateStatus())
            {
                return false;
            }

            return true;
        }

        bool SingleCommandInterface::FindFlagArgument(std::vector<FlagArgument>::iterator *pOut, std::string argument, std::vector<FlagArgument> &argumentDefinitions)
        {
            auto foundArgument = std::find_if(argumentDefinitions.begin(), argumentDefinitions.end(), [argument](FlagArgument &v){ return v.IsMatch(argument); });
            if (foundArgument != argumentDefinitions.end())
            {
                *pOut = foundArgument;
                return true;
            }

            return false;
        }

        bool SingleCommandInterface::FindKeywordArgument(std::vector<KeywordArgument>::iterator *pOut, std::string argument, std::vector<KeywordArgument> &argumentDefinitions)
        {
            auto foundArgument = std::find_if(argumentDefinitions.begin(), argumentDefinitions.end(), [argument](KeywordArgument &v){ return v.IsMatch(argument); });
            if (foundArgument != argumentDefinitions.end())
            {
                *pOut = foundArgument;
                return true;
            }

            return false;
        }

        bool SingleCommandInterface::FindPositionalArgument(std::vector<PositionalArgument>::iterator *pOut, int index, std::vector<PositionalArgument> &argumentDefinitions)
        {
            auto foundArgument = std::find_if(argumentDefinitions.begin(), argumentDefinitions.end(), [index](PositionalArgument &v) { return v.IsMatch(index); });
            if (foundArgument != argumentDefinitions.end())
            {
                *pOut = foundArgument;
                return true;
            }

            return false;
        }

    }
}
