﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_IntrusiveDic.h>

namespace nn { namespace util {

NN_DEFINE_STATIC_CONSTANT( const int IntrusiveDicNode::Npos );

IntrusiveDicNode* IntrusiveDicNode::Erase( IntrusiveDicNode** ppOther, int* pRefBit ) NN_NOEXCEPT
{
    int childIndex = ExtractRefBit( m_Key, m_RefBit );
    IntrusiveDicNode* pChild = m_pChildren[ childIndex ]; // 自身のキーによる子ノードを昇格させます。
    *ppOther = m_pChildren[ 1 - childIndex ]; // 反対の子ノードは昇格させたノードの子にします。
    *pRefBit = m_RefBit; // 参照ビットは子ノードに引き継ぎます。

    IntrusiveDicNode* pNode = pChild; // 実際に昇格させる子ノードです。
    if ( pChild == this )
    {
        // 参照ビット方向に昇格させる子ノードが無い場合は反対の子ノードを昇格させます。
        pNode = *ppOther;
    }

    if ( m_pParent->m_pChildren[ 0 ] == this ) // 自分と同じ方向の子ノードとします。
    {
        m_pParent->m_pChildren[ 0 ] = pNode;
    }
    else
    {
        m_pParent->m_pChildren[ 1 ] = pNode;
    }
    if ( IsOlderThan( pNode ) )
    {
        pNode->m_pParent = m_pParent;
    }

    Clear();
    return pChild;
}

IntrusiveDicNode* IntrusiveDicNode::InsertChild( IntrusiveDicNode** ppNode, int* pRefBit, const nn::util::string_view& key ) NN_NOEXCEPT
{
    IntrusiveDicNode* pNode = *ppNode; // 新たに子とするノードです。
    int refBit = *pRefBit; // 親ノードから移植する参照ビットです。

    int childIndex = ExtractRefBit( key, m_RefBit );
    IntrusiveDicNode* pChild = m_pChildren[ childIndex ]; // 子として残すノードです。
    *ppNode = m_pChildren[ 1 - childIndex ]; // 孫として移すノードです。
    *pRefBit = m_RefBit; // 参照ビットを子ノードに移植します。

    int selfIndex = ExtractRefBit( key, refBit ); // 親ノードが自分を指していた方向です。
    m_pChildren[ selfIndex ] = IsOlderThan( pChild ) ? pChild : *ppNode;
    m_pChildren[ 1 - selfIndex ] = pNode;
    m_RefBit = refBit;
    if ( IsOlderThan( pNode ) ) // 参照ビットを移植してから判定する必要があります。
    {
        pNode->m_pParent = this;
    }
    return pChild;
}

bool IntrusiveDicNode::FindRefBit( const IntrusiveDicNode* pNode ) NN_NOEXCEPT
{
    // 戻った先のノードエントリと違うビットを探します。
    // refBit ビット目で判断することにします。
    const nn::util::string_view& key = pNode->m_Key;

    int endBit = static_cast< int >( CHAR_BIT * std::max NN_PREVENT_MACRO_FUNC ( m_Key.length(), key.length() ) );
    for ( int refBit = 0; refBit < endBit; ++refBit )
    {
        if ( ExtractRefBit( m_Key, refBit ) != ExtractRefBit( key, refBit ) )
        {
            m_RefBit = refBit;
            return true;
        }
    }
    return false; // 名前が重複している場合は失敗です。
}

int IntrusiveDicNode::ExtractRefBit( const nn::util::string_view& key, int refBit ) NN_NOEXCEPT
{
    // 文字列の末尾から数えて refBit 番目のビットを返します。
    int charIndex = refBit >> detail::Log2< NN_BITSIZEOF( char ) >::Value;
    if ( static_cast< size_t >( charIndex ) < static_cast< size_t >( key.length() ) )
    {
        int bitIndex = refBit & ( NN_BITSIZEOF( char ) - 1 );
        return ( key[ key.length() - charIndex - 1 ] >> bitIndex ) & 0x1;
    }
    else
    {
        // 参照ビットが key の外の場合は 0 を返します。
        return 0;
    }
}

//--------------------------------------------------------------------------------------------------

namespace detail {

const IntrusiveDicNode* IntrusiveDicImpl::FindImpl( const nn::util::string_view& key ) const NN_NOEXCEPT
{
    const IntrusiveDicNode* pParent = &m_Root;
    const IntrusiveDicNode* pChild = pParent->GetChild( 0 );

    while ( pParent->IsOlderThan( pChild ) )
    {
        pParent = pChild;
        pChild = pChild->GetChild( key );
    }

    return pChild;
}

void IntrusiveDicImpl::InsertImpl( IntrusiveDicNode* pNode ) NN_NOEXCEPT
{
    IntrusiveDicNode* pParent = &m_Root;
    IntrusiveDicNode* pChild = pParent->GetChild( 0 );

    while ( pParent->IsOlderThan( pChild ) && pChild->IsOlderThan( pNode ) )
    {
        pParent = pChild;
        pChild = pChild->GetChild( pNode->GetKey() );
    }
    // この時点で pParent と pChild の間にノードを挿入することが確定しています。

    pParent->SetChild( pNode );
    pNode->SetChild( pChild );
}

void IntrusiveDicImpl::EraseImpl( IntrusiveDicNode* pNode ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pNode != &m_Root );

    // ノード内のキーは削除時にクリアするのでコピーしておきます。
    const nn::util::string_view key = pNode->GetKey();
    int refBit = 0;
    IntrusiveDicNode* pParent = pNode->GetParent();
    IntrusiveDicNode* pOther = NULL;
    IntrusiveDicNode* pChild = pNode->Erase( &pOther, &refBit ); // pNode の位置に pChild が昇格します。

    // pChild と反対側の子ノード pOther を pChild の子として挿入していきます。
    while ( pParent->IsOlderThan( pChild ) )
    {
        pParent = pChild;
        pChild = pParent->InsertChild( &pOther, &refBit, key );
    }
}

const IntrusiveDicNode* IntrusiveDicImpl::FindIter() const NN_NOEXCEPT
{
    const IntrusiveDicNode* pParent = &m_Root;
    const IntrusiveDicNode* pChild = pParent->GetChild( 0 );

    // ルートノードが見つかるまで左側の子ノードを辿ります。
    while ( pChild != &m_Root )
    {
        pParent = pChild;
        pChild = pParent->GetChild( 0 );
    }
    // ルートノードと逆の子ノードから左側のノードを辿ります。
    pChild = pParent->GetChild( 1 );
    while ( pParent->IsOlderThan( pChild ) )
    {
        pParent = pChild;
        pChild = pParent->GetChild( 0 );
    }

    return pChild;
}

const IntrusiveDicNode* IntrusiveDicImpl::MoveIter( const IntrusiveDicNode* pNode, int fwdIndex ) NN_NOEXCEPT
{
    int revIndex = 1 - fwdIndex;
    const IntrusiveDicNode* pParent = FindParent( pNode );
    const IntrusiveDicNode* pChild = pNode;

    // 進行方向が現在指している子供であれば、進行方向に子ノードが現れ無くなるまで親を辿ります。
    if ( pChild == pParent->GetChild( fwdIndex ) )
    {
        do
        {
            pChild = pParent;
            pParent = pParent->GetParent();
        } while ( pChild == pParent->GetChild( fwdIndex ) );
    }

    // 進行方向に1つ進めた後、進行方向と最も逆方向の子ノードを探索します。
    pChild = pParent->GetChild( fwdIndex );
    while ( pParent->IsOlderThan( pChild ) )
    {
        pParent = pChild;
        pChild = pChild->GetChild( revIndex );
    }

    return pChild;
}

const IntrusiveDicNode* IntrusiveDicImpl::FindParent( const IntrusiveDicNode* pNode ) NN_NOEXCEPT
{
    const nn::util::string_view& key = pNode->GetKey();
    const IntrusiveDicNode* pParent = pNode;
    const IntrusiveDicNode* pChild = pNode;

    // ノードのキーで辿ることにより自身にたどり着くことができます。
    do
    {
        pParent = pChild;
        pChild = pParent->GetChild( key );
    } while ( pParent->IsOlderThan( pChild ) );
    NN_SDK_ASSERT( pChild == pNode );

    return pParent;
}

}}} // namespace nn/util/detail
