﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BitArray.h>

namespace nn { namespace util {

BitArray& BitArray::set() NN_NOEXCEPT
{
    int numBlock_1 = GetBlockCount() - 1;

    if ( numBlock_1 >= 0 )
    {
        std::fill_n( m_pBuf, numBlock_1, RawType( -1 ) );
        m_pBuf[ numBlock_1 ] = MakeLastBlockBitMask();
    }

    return *this;
}

BitArray& BitArray::reset() NN_NOEXCEPT
{
    if ( int numBlock = GetBlockCount() )
    {
        std::fill_n( m_pBuf, numBlock, 0 );
    }
    return *this;
}

BitArray& BitArray::flip() NN_NOEXCEPT
{
    int numBlock_1 = GetBlockCount() - 1;

    if ( numBlock_1 >= 0 )
    {
        for (int i = 0; i < numBlock_1; ++i)
        {
            m_pBuf[ i ] = ~m_pBuf[ i ];
        }

        m_pBuf[ numBlock_1 ] = ~m_pBuf[ numBlock_1 ] & MakeLastBlockBitMask();
    }

    return *this;
}

BitArray::size_type BitArray::count() const NN_NOEXCEPT
{
    int numBlock = GetBlockCount();

    int sum = 0;
    for (int i = 0; i < numBlock; ++i)
    {
        sum += popcount( m_pBuf[ i ] );
    }

    return sum;
}

bool BitArray::none() const NN_NOEXCEPT
{
    int numBlock = GetBlockCount();

    for (int i = 0; i < numBlock; ++i)
    {
        if ( m_pBuf[ i ] != 0 )
        {
            return false;
        }
    }

    return true;
}

bool BitArray::all() const NN_NOEXCEPT
{
    int numBlock_1 = GetBlockCount() - 1;

    const RawType mask = RawType( -1 );
    for (int i = 0; i < numBlock_1; ++i)
    {
        if ( m_pBuf[ i ] != mask )
        {
            return false;
        }
    }

    if ( numBlock_1 < 0 )
    {
        return false;
    }

    return m_pBuf[ numBlock_1 ] == MakeLastBlockBitMask();
}

void BitArray::And( BitArray* pResult, const BitArray& lhs, const BitArray& rhs ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( lhs.size(), rhs.size() );
    NN_SDK_REQUIRES_EQUAL( pResult->size(), lhs.size() );

    int numBlock = lhs.GetBlockCount();
    for (int i = 0; i < numBlock; ++i)
    {
        pResult->m_pBuf[ i ] = lhs.m_pBuf[ i ] & rhs.m_pBuf[ i ];
    }
}

void BitArray::Or( BitArray* pResult, const BitArray& lhs, const BitArray& rhs ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( lhs.size(), rhs.size() );
    NN_SDK_REQUIRES_EQUAL( pResult->size(), lhs.size() );

    int numBlock = lhs.GetBlockCount();
    for (int i = 0; i < numBlock; ++i)
    {
        pResult->m_pBuf[ i ] = lhs.m_pBuf[ i ] | rhs.m_pBuf[ i ];
    }
}

void BitArray::Xor( BitArray* pResult, const BitArray& lhs, const BitArray& rhs ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( lhs.size(), rhs.size() );
    NN_SDK_REQUIRES_EQUAL( pResult->size(), lhs.size() );

    int numBlock_1 = lhs.GetBlockCount() - 1;
    for (int i = 0; i < numBlock_1; ++i)
    {
        pResult->m_pBuf[ i ] = lhs.m_pBuf[ i ] ^ rhs.m_pBuf[ i ];
    }

    if ( numBlock_1 >= 0 )
    {
        pResult->m_pBuf[ numBlock_1 ] = lhs.m_pBuf[ numBlock_1 ] ^ rhs.m_pBuf[ numBlock_1 ];
        pResult->m_pBuf[ numBlock_1 ] &= pResult->MakeLastBlockBitMask();
    }

}

void BitArray::Not( BitArray* pResult, const BitArray& lhs ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( pResult->size(), lhs.size() );

    int numBlock_1 = lhs.GetBlockCount() - 1;

    if ( numBlock_1 >= 0 )
    {
        for (int i = 0; i < numBlock_1; ++i)
        {
            pResult->m_pBuf[ i ] = ~lhs.m_pBuf[ i ];
        }

        pResult->m_pBuf[ numBlock_1 ] = ~lhs.m_pBuf[ numBlock_1 ] & lhs.MakeLastBlockBitMask();
    }
}

void BitArray::RightShift( BitArray* pResult, const BitArray& lhs, size_type pos ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( pResult->size(), lhs.size() );

    int numBlock = lhs.GetBlockCount();

    int quotient = pos >> BlockShift;
    int remainder = pos & static_cast< int >( BlockBitMask );

    for (int i = numBlock - 1; i > quotient; --i)
    {
        pResult->m_pBuf[ i ] = lhs.m_pBuf[ i - quotient ];

        if ( remainder )
        {
            pResult->m_pBuf[ i ] <<= remainder;
            pResult->m_pBuf[ i ] |= ( ( lhs.m_pBuf[ i - quotient - 1 ] & align_down( RawType( -1 ), RawType( 1 ) << ( BitsPerBlock - remainder ) ) )
                >> ( BitsPerBlock - remainder ) );
        }
    }

    if ( numBlock > quotient )
    {
        pResult->m_pBuf[ quotient ] = lhs.m_pBuf[ 0 ];
        pResult->m_pBuf[ quotient ] <<= remainder;
    }

    int emptyBlockCount = std::min( numBlock, quotient );
    for (int i = 0; i < emptyBlockCount; ++i)
    {
        pResult->m_pBuf[ i ] = RawType( 0 );
    }
}

void BitArray::LeftShift( BitArray* pResult, const BitArray& lhs, size_type pos ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( pResult->size(), lhs.size() );

    int numBlock = lhs.GetBlockCount();

    int quotient = pos >> BlockShift;
    int remainder = pos & static_cast< int >( BlockBitMask );

    for (int i = 0; i < numBlock - quotient - 1; ++i)
    {
        pResult->m_pBuf[ i ] = lhs.m_pBuf[ i + quotient ];

        if ( remainder )
        {
            pResult->m_pBuf[ i ] >>= remainder;
            pResult->m_pBuf[ i ] |= ( ( lhs.m_pBuf[ i + quotient + 1 ] & ( ( RawType( 1 ) << remainder ) - 1 ) ) << ( BitsPerBlock - remainder ) );
        }
    }

    if ( numBlock > quotient )
    {
        pResult->m_pBuf[ numBlock - quotient - 1 ] = lhs.m_pBuf[ numBlock - 1 ];
        pResult->m_pBuf[ numBlock - quotient - 1 ] >>= remainder;
    }

    int emptyBlockCount = std::min( numBlock, quotient );
    for (int i = numBlock - emptyBlockCount; i < numBlock; ++i)
    {
        pResult->m_pBuf[ i ] = RawType( 0 );
    }
}

bool BitArray::operator==(const BitArray& rhs) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL( m_Len, rhs.m_Len );

    int numBlock = GetBlockCount();
    for (int i = 0; i < numBlock; ++i)
    {
        if ( m_pBuf[ i ] != rhs.m_pBuf[ i ] )
        {
            return false;
        }
    }

    return true;
}

}} // namespace nn::util
