﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_ExifBuilder.h>

#include "detail/image_JpegConfig.h"
#include "detail/image_LibjpegErrorHandler.h"
#include "detail/image_JpegParser.h"
#include "detail/image_LibjpegHelper.h"

#include "include/jpeglib.h"

#include <cstddef>
#include <csetjmp>
#include <cstring>
#include <climits>
#include <algorithm>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace image {

namespace
{
/**
    @brief JPEG エンコードのためのパラメータ設定
 */
void SetupEncodingParameter(
    detail::jpeg::jpeg_compress_struct *pCinfo,
    const unsigned int kQuality,
    const JpegSamplingRatio kSample,
    const bool kIsJfif) NN_NOEXCEPT
{
    // 入力データの情報
    pCinfo->input_components  = detail::LibjpegProperty_BytesPerPx;
    pCinfo->in_color_space    = detail::jpeg::JCS_RGB;

    // 出力色空間と、サンプリング比を設定
    detail::jpeg::jpeg_set_defaults(pCinfo);
    detail::jpeg::jpeg_set_colorspace(pCinfo, detail::jpeg::JCS_YCbCr);
    switch (kSample)
    {
    case JpegSamplingRatio_444:
        pCinfo->comp_info[0].h_samp_factor = 1;
        pCinfo->comp_info[0].v_samp_factor = 1;
        pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
        pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;
        break;
    case JpegSamplingRatio_422:
        pCinfo->comp_info[0].h_samp_factor = 2;
        pCinfo->comp_info[0].v_samp_factor = 1;
        pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
        pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;
        break;
    case JpegSamplingRatio_420:
        pCinfo->comp_info[0].h_samp_factor = 2;
        pCinfo->comp_info[0].v_samp_factor = 2;
        pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
        pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 画質
    detail::jpeg::jpeg_set_quality(pCinfo, kQuality, true);

    // エンコード画質と処理速度のパラメータ
    pCinfo->dct_method = detail::jpeg::JDCT_ISLOW;
    pCinfo->optimize_coding = false;
    pCinfo->do_fancy_downsampling = false;

    // もし Exif 情報を出力するなら、JFIF セグメントは記録しないようにする。
    // (Exif の仕様)
    if (!kIsJfif)
    {
        pCinfo->write_JFIF_header = false;
    }
}

void EncodeRgba32(
    detail::jpeg::jpeg_compress_struct& cinfo,
    const uint8_t* pxData,
    const Dimension kDim,
    const int kLineAlignment,
    void* pxWorkBuf,
    size_t pxWorkBufSize) NN_NOEXCEPT
{
    NN_UNUSED(pxWorkBufSize);

    // ピクセルデータのコピー用のバッファ(RGB)を用意
    const auto kRowStride = kDim.width * detail::LibjpegProperty_BytesPerPx;
    detail::jpeg::JSAMPROW lines[detail::LibjpegProperty_1LoopLineNum] = {};
    for (int i = 0; i < detail::LibjpegProperty_1LoopLineNum; i++)
    {
        lines[i] = reinterpret_cast<detail::jpeg::JSAMPLE*>(pxWorkBuf) + i * kRowStride;
    }

    auto alignedWidth = detail::RoundUp(kDim.width, kLineAlignment);
    while (cinfo.next_scanline < kDim.height)
    {
        // 最大 LibjpegProperty_1LoopLineNum 行処理する
        auto currentLine = cinfo.next_scanline;
        auto linesToEncode = std::min<int32_t>(kDim.height - currentLine, detail::LibjpegProperty_1LoopLineNum);

        // 入力の RGBA データを RGB のバッファに格納する
        for (auto i = 0; i < linesToEncode; i++)
        {
            const uint8_t *pixelLine = pxData + (currentLine + i) * alignedWidth * sizeof(nn::Bit32);

            detail::jpeg::JSAMPLE *line = lines[i];
            for (uint16_t j = 0; j < kDim.width; j++, line += detail::LibjpegProperty_BytesPerPx)
            {
                int pixelIndex = j * sizeof(nn::Bit32);
                line[0] = (detail::jpeg::JSAMPLE)(pixelLine[pixelIndex + 0]); // R
                line[1] = (detail::jpeg::JSAMPLE)(pixelLine[pixelIndex + 1]); // G
                line[2] = (detail::jpeg::JSAMPLE)(pixelLine[pixelIndex + 2]); // B
            }
        }

        // エンコード処理 = write_scanlines
        detail::jpeg::jpeg_write_scanlines(&cinfo, lines, linesToEncode);
    }
}

void EncodeRgb24(
    detail::jpeg::jpeg_compress_struct& cinfo,
    const uint8_t* pxData,
    const Dimension kDim,
    const int kLineAlignment) NN_NOEXCEPT
{
    const auto kRowStride = detail::RoundUp(kDim.width, kLineAlignment) * detail::LibjpegProperty_BytesPerPx;

    detail::jpeg::JSAMPROW lines[detail::LibjpegProperty_1LoopLineNum];
    while (cinfo.next_scanline < kDim.height)
    {
        // 最大 LibjpegProperty_1LoopLineNum 行処理する
        auto currentLine = cinfo.next_scanline;
        auto linesToEncode = std::min<int32_t>(kDim.height - currentLine, detail::LibjpegProperty_1LoopLineNum);

        for (auto i = 0; i < linesToEncode; i++)
        {
            lines[i] = reinterpret_cast<detail::jpeg::JSAMPLE*>(const_cast<uint8_t*>(pxData) + (currentLine + i) * kRowStride);
        }

        // エンコード処理 = write_scanlines
        detail::jpeg::jpeg_write_scanlines(&cinfo, lines, linesToEncode);
    }
}

/**
    @brief エンコードの実処理
 */
JpegStatus EncodeImpl(
    size_t *pActualCodedSize,
    void *jpegBuf,
    const size_t kJpegBufSize,
    const uint8_t *pixelData,
    const PixelFormat kPixelFormat,
    const ::nn::image::Dimension &kDim,
    const int kLineAlignment,
    const int kQuality,
    const JpegSamplingRatio kSample,
    const bool kIsJfif,
    void *workBuf,
    const size_t kWorkBufSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        reinterpret_cast<uintptr_t>(workBuf) % NN_ALIGNOF(std::max_align_t) == 0,
        "Invalid work buffer Alignment.\n");

    auto pxWorkBufSize = detail::GetPixelWorkBufferSize(kDim.width, kPixelFormat);
    NN_ABORT_UNLESS(pxWorkBufSize < kWorkBufSize);
    void* pxWorkBuf = pxWorkBufSize > 0
        ? reinterpret_cast<uint8_t*>(workBuf) + (kWorkBufSize - pxWorkBufSize)
        : nullptr;

    // libjpeg のインスタンスを作成
    detail::jpeg::jpeg_compress_struct cinfo = {};
    detail::jpeg::jpeg_workbuf wbMgr = {workBuf, kWorkBufSize - pxWorkBufSize};
    cinfo.workbuf = &wbMgr;

    // libjpeg のエラー処理の設定
    detail::LibjpegErrorInfo errInfo;
    // libjpeg でのエラー管理のための pub メンバを Compress Info に登録
    cinfo.err = detail::jpeg::jpeg_std_error(&errInfo);
    errInfo.error_exit = detail::HandleLibjpegError;
    errInfo.status = JpegStatus_Ok;

    // libjpeg 初期化
    detail::jpeg::jpeg_create_compress(&cinfo);

    JpegStatus rErrorCode = JpegStatus_Ok;
    // libjpeg 内部でのエラー発生時には setjmp に非 0 が返る。
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
    if (setjmp(errInfo.jmpContext) == 0)
    {
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        // 入力するピクセルデータをセット
        NN_SDK_ASSERT(kJpegBufSize <= ULONG_MAX);
        unsigned long codedSize = static_cast<decltype(codedSize)>(kJpegBufSize); // NOLINT(type/buildin)
        detail::jpeg::jpeg_mem_dest(&cinfo, reinterpret_cast<uint8_t**>(&jpegBuf), &codedSize);
        cinfo.image_width = kDim.width;
        cinfo.image_height = kDim.height;

        SetupEncodingParameter(&cinfo, kQuality, kSample, kIsJfif);

        // エンコード開始
        detail::jpeg::jpeg_start_compress(&cinfo, true);

        switch (kPixelFormat)
        {
        case PixelFormat_Rgba32:
            EncodeRgba32(cinfo, pixelData, kDim, kLineAlignment, pxWorkBuf, pxWorkBufSize);
            break;
        case PixelFormat_Rgb24:
            EncodeRgb24(cinfo, pixelData, kDim, kLineAlignment);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // エンコードの終了処理
        detail::jpeg::jpeg_finish_compress(&cinfo);
        *pActualCodedSize = codedSize;
    }
    else
    {
        NN_SDK_ASSERT(errInfo.status != JpegStatus_Ok, "Unexpected error code\n");
        rErrorCode = errInfo.status;
    }

    // libjpeg インスタンスの破棄
    detail::jpeg::jpeg_destroy_compress(&cinfo);
    return rErrorCode;
}
} // ~namespace nn::image::<anonymous>

JpegEncoder::JpegEncoder() NN_NOEXCEPT :
    m_Stage(Stage_Unregistered),
    m_Quality(100u),
    m_SamplingRatio(JpegSamplingRatio_420)
{
}

JpegEncoder::~JpegEncoder() NN_NOEXCEPT {}

void JpegEncoder::SetPixelData(
    const void *pixelData,
    const PixelFormat kPixelFormat,
    const ::nn::image::Dimension &kDimension,
    const int kLineAlignment) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(kLineAlignment >= 1, "Alignment must be >= 1\n");
    NN_SDK_REQUIRES(
        pixelData != nullptr && kDimension.width > 0 && kDimension.height > 0,
        "Requires valid pixel data\n");
    switch (kPixelFormat)
    {
    case PixelFormat_Rgba32:
    case PixelFormat_Rgb24:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_PixelData = pixelData;
    m_PixelFormat = kPixelFormat;
    m_Dim = kDimension;
    m_LineAlignment = kLineAlignment;

    m_Stage = Stage_Registered;
}

void JpegEncoder::SetQuality(const int kQuality) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        detail::CheckParamCompQuality(kQuality),
        "Encoding quality must be in [1..100]\n");

    m_Quality = kQuality;

    // ピクセルデータが設定されているとき、Analyzed であっても Registered に戻す。
    // そうでなければ変更しない。(Unregisterd を明示)
    m_Stage = ((m_PixelData != nullptr)? Stage_Registered: Stage_Unregistered);
}

void JpegEncoder::SetSamplingRatio(const JpegSamplingRatio kJpegSamplingRatio) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        kJpegSamplingRatio == JpegSamplingRatio_420 ||
        kJpegSamplingRatio == JpegSamplingRatio_422 ||
        kJpegSamplingRatio == JpegSamplingRatio_444,
        "Sampling ratio must be in {4:2:0, 4:2:2, 4:4:4}\n");
    m_SamplingRatio = kJpegSamplingRatio;

    // ピクセルデータが設定されているとき、Analyzed であっても Registered に戻す。
    // そうでなければ変更しない。(Unregisterd を明示)
    m_Stage = ((m_PixelData != nullptr)? Stage_Registered: Stage_Unregistered);
}

JpegStatus JpegEncoder::Analyze() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Stage >= Stage_Registered, "Pixel data is not registered.\n");

    // ピクセルデータのサイズと、サンプリング比から消費ワークメモリを計算
    size_t bufSize = 0u;
    JpegStatus s = detail::GetBufferSizeForJpegCompression(&bufSize, m_Dim, m_SamplingRatio);
    if (s != JpegStatus_Ok)
    {
        return s;
    }
    // 消費メモリ量に、ピクセルデータのコピー用のバッファサイズを上乗せ
    bufSize += detail::GetPixelWorkBufferSize(m_Dim.width, m_PixelFormat);

    // メンバ変数を更新
    m_BufSize = bufSize;
    m_Stage = Stage_Analyzed;
    return JpegStatus_Ok;
}

size_t JpegEncoder::GetAnalyzedWorkBufferSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Pixel data is not analyzed.\n");
    return m_BufSize;
}

JpegStatus JpegEncoder::Encode(
    size_t *pActualCodedSize,
    void *jpegBuf,
    const size_t kJpegBufSize,
    void *workBuf,
    const size_t kWorkBufSize,
    ExifBuilder *pExifBuilder) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Stage >= Stage_Analyzed, "Pixel data is not analyzed.\n");

    NN_SDK_REQUIRES(
        pActualCodedSize != nullptr &&
        jpegBuf != nullptr && kJpegBufSize > 0,
        "Valid pixel buffer is required.\n");
    NN_SDK_REQUIRES(
        workBuf != nullptr && kWorkBufSize >= m_BufSize,
        "Valid work buffer is required.\n");
    NN_SDK_REQUIRES(
        pExifBuilder == nullptr || pExifBuilder->IsBuildable(),
        "Valid ExifBuilder object is required if specified.\n");

    auto jpegBufSize = kJpegBufSize;
    void *app1Buf = nullptr;
    size_t app1Size = 0u;
    if (pExifBuilder != nullptr)
    {
        // Exif 情報を書きだす場合、出力バッファの先頭を JPEG 用に予約する
        app1Buf = jpegBuf;
        app1Size = pExifBuilder->GetAnalyzedOutputSize() + detail::JpegProperty_ExifAppHeaderSize;
        if (!(app1Size < kJpegBufSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Output buffer is too short (EXIF consumed all)\n");
            return JpegStatus_ShortOutput;
        }

        // JPEG 用の領域を削る
        jpegBufSize = kJpegBufSize - app1Size;
        jpegBuf = reinterpret_cast<uint8_t*>(jpegBuf) + app1Size;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO("APP1 consumed output buffer for %ubytes\n", app1Size);
    }

    // エンコード処理
    JpegStatus rErrorCode = EncodeImpl(
        pActualCodedSize,
        jpegBuf, jpegBufSize,
        reinterpret_cast<const uint8_t*>(m_PixelData),
        m_PixelFormat, m_Dim, m_LineAlignment,
        m_Quality, m_SamplingRatio,
        pExifBuilder == nullptr,
        workBuf, kWorkBufSize);
    if (rErrorCode != JpegStatus_Ok)
    {
        return rErrorCode;
    }

    if (app1Buf != nullptr)
    {
        // Exif 情報を書きだすのは、JPEG の SOI (0xFFD8) の直後
        uint8_t *out = reinterpret_cast<uint8_t*>(app1Buf);
        NN_SDK_ASSERT(
            out[app1Size] == 0xFF && out[app1Size + 1] == 0xD8,
            "Unexpected marker generated by libjpeg: %02x%02x\n",
            out[app1Size], out[app1Size + 1]);

        // まずは SOI を JPEG 用領域の先頭から、バッファ全体の先頭にコピー
        // (「SOI -> <空の app1Size 分の領域> -> SOI より後ろの JPEG データ」となっている。)
        std::memcpy(out, out + app1Size, 2);

        // SOI の直後に Exif 情報を出力する
        detail::WriteExifHeader(out + 2, app1Size);
        pExifBuilder->Build(
            out + 2 + detail::JpegProperty_ExifAppHeaderSize,
            pExifBuilder->GetAnalyzedOutputSize(),
            m_Dim);

        // JPEG データの全体の大きさに、Exif 情報の分を上乗せする
        *pActualCodedSize += app1Size;
    }

    return JpegStatus_Ok;
}

}}
