﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/edid/edid_DisplayModeInfo.h>
#include <nn/edid/edid_DisplayTimingInfo.h>
#include <nn/edid/edid_ImageSizeInfo.h>
#include "edid_DisplayModeInfo.h"
#include "edid_Serialize.h"
#include "../cea861/edid_Cea861.h"

const std::uint8_t AspectRatioMask = 0xC0;

const std::uint8_t VsyncPolarityMask = 0x1C;
const std::uint8_t HsyncPolarityMask = 0x12;

const std::uint8_t RefreshRateMask = 0x3F;

const std::uint8_t InterlacedMask  = 0x80;

const std::uint8_t AnalogSerrationMask  = 0x14;
const std::uint8_t DigitalSerrationMask = 0x1C;

const std::uint8_t CompositeSyncMask = 0x18;
const std::uint8_t SyncOnColorMask   = 0x12;

// MSb is "native" flag
const std::uint8_t VicMask = 0x7F;

void nn::edid::detail::VisitEstablishedTiming(nn::edid::DisplayModeVisitor visitor, void* pUserData, int width, int height, float refreshRate, bool isInterlaced) NN_NOEXCEPT
{
    nn::edid::DisplayModeInfo mode;

    mode.width = width;
    mode.height = height;
    mode.refreshRate = refreshRate;
    mode.imageAspectRatio = static_cast<float>(width) / height;
    mode.pixelAspectRatio = 1.f;
    mode.stereoMode = nn::edid::StereoMode_None;
    mode.isInterlaced = isInterlaced;

    visitor(&mode, nullptr, nullptr, pUserData);
}

void nn::edid::detail::GetEstablishedTimingI(nn::edid::DisplayModeVisitor visitor, void* pUserData, std::uint8_t timings) NN_NOEXCEPT
{
    // individual bits map to timings
    // See Table 3.18 in EDID 1.4 spec

    if( (timings & 0x01) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 800, 600, 60.00f, false);
    }

    if( (timings & 0x02) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 800, 600, 56.00f, false);
    }

    if( (timings & 0x04) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 640, 480, 75.00f, false);
    }

    if( (timings & 0x08) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 640, 480, 72.00f, false);
    }

    if( (timings & 0x10) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 640, 480, 67.00f, false);
    }

    if( (timings & 0x20) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 640, 480, 60.00f, false);
    }

    if( (timings & 0x40) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 720, 400, 88.00f, false);
    }

    if( (timings & 0x80) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 720, 400, 70.00f, false);
    }
}

void nn::edid::detail::GetEstablishedTimingII(nn::edid::DisplayModeVisitor visitor, void* pUserData, std::uint8_t timings) NN_NOEXCEPT
{
    // individual bits map to timings
    // See Table 3.18 in EDID 1.4 spec

    if( (timings & 0x01) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 1280, 1024, 75.00f, false);
    }

    if( (timings & 0x02) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 1024, 768, 75.00f, false);
    }

    if( (timings & 0x04) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 1024, 768, 70.00f, false);
    }

    if( (timings & 0x08) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 1024, 768, 60.00f, false);
    }

    if( (timings & 0x10) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 1024, 768, 87.00f, true);
    }

    if( (timings & 0x20) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 832, 624, 75.00f, false);
    }

    if( (timings & 0x40) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 800, 600, 75.00f, false);
    }

    if( (timings & 0x80) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 800, 600, 72.00f, false);
    }
}

void nn::edid::detail::GetEstablishedTimingIII(nn::edid::DisplayModeVisitor visitor, void* pUserData, std::uint8_t timings) NN_NOEXCEPT
{
    // individual bits map to timings
    // See Table 3.18 in EDID 1.4 spec

    if( (timings & 0x80) != 0 )
    {
        VisitEstablishedTiming(visitor, pUserData, 1152, 870, 75.00f, false);
    }

    // the rest of the timings in byte 3 are manufacturer specific
}

void nn::edid::detail::GetStandardTiming(DisplayModeInfo* pOutMode, std::uint8_t pixels, std::uint8_t timing) NN_NOEXCEPT
{
    // See Table 3.19 in EDID 1.4 spec for details
    NN_SDK_ASSERT_NOT_NULL(pOutMode);

    pOutMode->width = (pixels + 31) * 8;

    int aspectRatioWidth;
    int aspectRatioHeight;
    GetStandardTimingAspectRatio(&aspectRatioWidth, &aspectRatioHeight, timing);

    pOutMode->height = (pOutMode->width * aspectRatioWidth) / aspectRatioHeight;
    pOutMode->imageAspectRatio = static_cast<float>(aspectRatioWidth) / aspectRatioHeight;
    pOutMode->pixelAspectRatio = 1.f;
    pOutMode->stereoMode = StereoMode_None;
    pOutMode->refreshRate = (timing & RefreshRateMask) + 60.00f;

    // TODO: is this correct?
    pOutMode->isInterlaced = false;
}

void nn::edid::detail::GetStandardTimingAspectRatio(int* pOutWidth, int* pOutHeight, std::uint8_t timing) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutWidth);
    NN_SDK_ASSERT_NOT_NULL(pOutHeight);

    switch( timing & AspectRatioMask )
    {
    case 0x00:
        *pOutWidth = 16;
        *pOutHeight = 10;
        break;
    case 0x40:
        *pOutWidth = 4;
        *pOutHeight = 3;
        break;
    case 0x80:
        *pOutWidth = 5;
        *pOutHeight = 4;
        break;
    case 0xC0:
        *pOutWidth = 16;
        *pOutHeight = 9;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::edid::Polarity nn::edid::detail::GetDetailedTimingVsyncPolarity(std::uint8_t features) NN_NOEXCEPT
{
    // bit 7 must be clear -- only for digital interfaces
    switch( features & VsyncPolarityMask )
    {
    case 0x18:
        return Polarity_Negative;
    case 0x1C:
        return Polarity_Positive;
    default:
        return Polarity_Undefined;
    }
}

nn::edid::Polarity nn::edid::detail::GetDetailedTimingHsyncPolarity(std::uint8_t features) NN_NOEXCEPT
{
    // bit 7 must be clear -- only for digital interfaces
    switch( features & HsyncPolarityMask )
    {
    case 0x10:
        return Polarity_Negative;
    case 0x12:
        return Polarity_Positive;
    default:
        return Polarity_Undefined;
    }
}

nn::edid::StereoMode nn::edid::detail::GetDetailedTimingStereoMode(std::uint8_t features) NN_NOEXCEPT
{
    // Check Table 3.22 in EDID 1.4 spec

    switch( features & 0x61 )
    {
    case 0x00:
        NN_FALL_THROUGH;
        // odd case since bit 0 is defined as "don't care"
    case 0x01:
        return StereoMode_None;
    case 0x20:
        return StereoMode_FieldSequentialRight;
    case 0x40:
        return StereoMode_FieldSequentialLeft;
    case 0x21:
        return StereoMode_InterleavedRight;
    case 0x41:
        return StereoMode_InterleavedLeft;
    case 0x60:
        return StereoMode_Interleaved;
    case 0x61:
        return StereoMode_SideBySide;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::edid::SyncTypeSet nn::edid::detail::GetDetailedTimingSyncTypes(std::uint8_t features) NN_NOEXCEPT
{
    SyncTypeSet types;
    types.Reset();

    // odd case since this applies to both analog and digital
    types.Set<SyncType::CompositeSync>((features & CompositeSyncMask) == 0 || (features & CompositeSyncMask) == 0x10);

    // analog only
    types.Set<SyncType::BipolarCompositeSync>((features & CompositeSyncMask) == 8);
    types.Set<SyncType::SyncOnGreen>((features & SyncOnColorMask) == 0);
    types.Set<SyncType::SyncOnRgb>((features & SyncOnColorMask) == 2);

    // digital only
    types.Set<SyncType::SeparateSync>((features & CompositeSyncMask) == 0x18);

    return types;
}

bool nn::edid::detail::IsDetailedTimingSerrated(std::uint8_t features) NN_NOEXCEPT
{
    return ((features & AnalogSerrationMask) == 4) || ((features & DigitalSerrationMask) == 0x14);
}

bool nn::edid::detail::IsDetailedTimingInterlaced(std::uint8_t features) NN_NOEXCEPT
{
    return (features & InterlacedMask) != 0;
}

bool nn::edid::detail::GetDetailedTiming(DisplayModeInfo* pOutMode, DisplayTimingInfo* pOutTiming, ImageSizeInfo* pOutImage, const std::uint8_t* pDetailedTiming, size_t size) NN_NOEXCEPT
{
    // See Tables 3.21, 3.22 in EDID 1.4 spec
    NN_ABORT_UNLESS(size == detail::DetailedTimingStride);

    if( *pDetailedTiming == 0x00 && pDetailedTiming[1] == 0x00 )
    {
        // this indicates this section is a display descriptor and not a DTD
        return false;
    }

    // converting stored value to kHz
    pOutTiming->pixelClock = detail::Serialize<std::uint16_t>(pDetailedTiming) * 10;

    // these are 12 bit values
    pOutMode->width = pDetailedTiming[0x02] | ((pDetailedTiming[0x04] & 0xF0) << 4);
    pOutTiming->hactive = static_cast<std::uint16_t>(pOutMode->width);
    pOutTiming->hblank = pDetailedTiming[0x03] | ((pDetailedTiming[0x04] & 0x0F) << 8);
    pOutMode->height = pDetailedTiming[0x05] | ((pDetailedTiming[0x07] & 0xF0) << 4);
    pOutTiming->vactive = static_cast<std::uint16_t>(pOutMode->height);
    pOutTiming->vblank = pDetailedTiming[0x06] | ((pDetailedTiming[0x07] & 0x0F) << 8);

    // these are 10 bit values
    pOutTiming->horizontalFrontPorch = pDetailedTiming[0x08] | ((pDetailedTiming[0x0B] & 0xC0) << 2);
    pOutTiming->horizontalSyncPulse = pDetailedTiming[0x09] | ((pDetailedTiming[0x0B] & 0x30) << 4);

    // these are 6 bit values
    pOutTiming->verticalFrontPorch = ((pDetailedTiming[0x0A] & 0xF0) >> 4) | ((pDetailedTiming[0x0B] & 0x0C) << 2);
    pOutTiming->verticalSyncPulse = (pDetailedTiming[0x0A] & 0x0F) | ((pDetailedTiming[0x0B] & 0x03) << 4);

    // more 12 bit values
    pOutImage->imageWidth = pDetailedTiming[0x0C] | ((pDetailedTiming[0x0E] & 0xF0) << 4);
    pOutImage->imageHeight = pDetailedTiming[0x0D] | ((pDetailedTiming[0x0E] & 0x0F) << 8);

    pOutImage->horizontalBorder = pDetailedTiming[0x0F];
    pOutImage->verticalBorder = pDetailedTiming[0x10];

    std::uint8_t features = pDetailedTiming[0x11];
    pOutMode->isInterlaced = detail::IsDetailedTimingInterlaced(features);

    pOutMode->stereoMode = detail::GetDetailedTimingStereoMode(features);

    pOutTiming->vsyncPolarity = detail::GetDetailedTimingVsyncPolarity(features);
    pOutTiming->hsyncPolarity = detail::GetDetailedTimingHsyncPolarity(features);

    pOutTiming->hasSyncSerrations = detail::IsDetailedTimingSerrated(features);

    pOutTiming->analogSync = detail::GetDetailedTimingSyncTypes(features);

    if( pOutMode->isInterlaced )
    {
        pOutMode->height *= 2;
    }

    // derived values
    if( pOutTiming->hactive + pOutTiming->hblank != 0 &&
        pOutTiming->vactive + pOutTiming->vblank != 0 )
    {
        pOutMode->refreshRate = ((pOutTiming->pixelClock * 1000.f) / (pOutTiming->hactive + pOutTiming->hblank)) / (pOutTiming->vactive + pOutTiming->vblank);
    }
    else
    {
        pOutMode->refreshRate = 0.f;
    }

    if( pOutMode->height != 0 )
    {
        pOutMode->imageAspectRatio = static_cast<float>(pOutMode->width) / pOutMode->height;
    }
    else
    {
        pOutMode->imageAspectRatio = 0.f;
    }

    if( pOutImage->imageHeight * pOutMode->width != 0 )
    {
        pOutMode->pixelAspectRatio = static_cast<float>(pOutImage->imageWidth * pOutMode->height) / (pOutImage->imageHeight * pOutMode->width);
    }
    else
    {
        pOutMode->pixelAspectRatio = 0.f;
    }

    return true;
}

bool nn::edid::detail::DetailedTimingVisitor(const DisplayModeInfo* pMode, const DisplayTimingInfo* pTiming, const ImageSizeInfo* pImage, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    DetailedTimingVisitorData* pData = static_cast<DetailedTimingVisitorData*>(pUserData);

    return pData->visitor(pMode, pTiming, pImage, pData->pUserData);
}

bool nn::edid::detail::ModeExtensionBlockVisitor(const std::uint8_t* pBlock, size_t size, void* pUserData) NN_NOEXCEPT
{
    cea861::VisitDataBlocks(pBlock, size, VideoBlockVisitor, pUserData);
    cea861::VisitDetailedTimings(pBlock, size, DetailedTimingVisitor, pUserData);

    return true;
}

bool nn::edid::detail::VideoBlockVisitor(const std::uint8_t* pBlock, size_t size, cea861::BlockTag tag, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    DetailedTimingVisitorData* pData = static_cast<DetailedTimingVisitorData*>(pUserData);

    switch( tag )
    {
    case cea861::BlockTag::Video:
        for( size_t i = 1; i < size; ++i )
        {
            DisplayModeInfo mode;

            if( cea861::ConvertVic(&mode, pBlock[i] & VicMask) )
            {
                pData->visitor(&mode, nullptr, nullptr, pData->pUserData);
            }
        }
        break;
    default:
        // Do nothing -- fixes warning for clang
        break;
    }

    return true;
}
