﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/edid/edid_DisplayInfo.h>
#include <nn/nn_SdkAssert.h>
#include "edid_DisplayInfo.h"
#include "edid_Serialize.h"

const std::uint16_t FirstCharacterMask  = 0x7C00;
const std::uint16_t SecondCharacterMask = 0x3E0;
const std::uint16_t ThirdCharacterMask  = 0x001F;

const std::uint8_t  VideoInterfaceAnalogMask  = 0x80;
const std::uint8_t  VideoInterfaceDigitalMask = 0x0F;

const std::uint8_t  SignalLevelMask  = 0xE0;
// need bit 7 (analog/digital) and bit 4
const std::uint8_t  BlankLevelMask    = 0x90;
const std::uint8_t  SerrationsMask    = 0x81;
const std::uint8_t  ColorDepthMask    = 0xF0;
const std::uint8_t  ColorSupportMask  = 0x18;

const std::uint8_t  SeparateSyncMask  = 0x88;
const std::uint8_t  CompositeSyncMask = 0x84;
const std::uint8_t  SyncOnGreenMask   = 0x82;

const std::uint8_t  YccQuantizationMask = 0x80;
const std::uint8_t  RgbQuantizationMask = 0x40;

const std::uint8_t  GraphicsTypeMask = 0x01;
const std::uint8_t  PhotoTypeMask    = 0x02;
const std::uint8_t  CinemaTypeMask   = 0x04;
const std::uint8_t  GameTypeMask     = 0x08;

const std::uint8_t  StandbyModeMask         = 0x80;
const std::uint8_t  SuspendModeMask         = 0x40;
const std::uint8_t  VeryLowPowerMask        = 0x20;
const std::uint8_t  SrgbColorMask           = 0x04;
const std::uint8_t  ContinuousFrequencyMask = 0x01;

void nn::edid::detail::Initialize(DisplayInfo* pOutDisplay) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutDisplay);

    std::memset(pOutDisplay, 0, sizeof(DisplayInfo));

    pOutDisplay->interfaceType = VideoInterface_Digital;
    pOutDisplay->sync = SyncLevel_Digital;
    pOutDisplay->blankType = BlankLevel_Digital;
    pOutDisplay->depth = ColorDepth_Undefined;
    pOutDisplay->colorSupport = ColorType_Undefined;
}

char nn::edid::detail::ToAscii(std::uint8_t value) NN_NOEXCEPT
{
    // ID manufacturer name uses 5-bit ASCII codes to represent A-Z
    if( value == 0 || value > 26 )
    {
        return 0;
    }

    return static_cast<char>(value) + 'A' - 1;
}

void nn::edid::detail::GetManufacturerCode(DisplayInfo* pOutDisplay, std::uint16_t value) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(sizeof(pOutDisplay->manufacturerCode) == 4);

    //byte swap for little endian
    //TODO:support big endian platform
    value = (((value & 0xff) << 8) | ((value >> 8) & 0xff));

    pOutDisplay->manufacturerCode[0] = detail::ToAscii((value & FirstCharacterMask) >> 10);
    pOutDisplay->manufacturerCode[1] = detail::ToAscii(static_cast<std::uint8_t>((value & SecondCharacterMask) >> 5));
    pOutDisplay->manufacturerCode[2] = detail::ToAscii(value & ThirdCharacterMask);
    pOutDisplay->manufacturerCode[3] = 0;
}

nn::edid::VideoInterface nn::edid::detail::GetVideoInterface(std::uint8_t value) NN_NOEXCEPT
{
    if( (value & VideoInterfaceAnalogMask) == 0 )
    {
        return VideoInterface_Analog;
    }

    // interface must be digital
    switch( value & VideoInterfaceDigitalMask )
    {
    case 0x01:
        return VideoInterface_Dvi;
    case 0x02:
        return VideoInterface_HdmiA;
    case 0x03:
        return VideoInterface_HdmiB;
    case 0x04:
        return VideoInterface_Mddi;
    case 0x05:
        return VideoInterface_DisplayPort;
    default:
        return VideoInterface_Digital;
    }
}

nn::edid::SyncLevel nn::edid::detail::GetSignalLevel(std::uint8_t value) NN_NOEXCEPT
{
    // bit 7 determines analog/digital
    // bits 5 and 6 define signal level standard
    switch( value & SignalLevelMask )
    {
    case 0x00:
        return SyncLevel_300;
    case 0x20:
        return SyncLevel_286;
    case 0x40:
        return SyncLevel_400;
    case 0x60:
        return SyncLevel_000;
    default:
        return SyncLevel_Digital;
    }
}

nn::edid::BlankLevel nn::edid::detail::GetVideoSetup(std::uint8_t value) NN_NOEXCEPT
{
    switch( value & BlankLevelMask )
    {
    case 0x00:
        return BlankLevel_Equal;
    case 0x10:
        return BlankLevel_Setup;
    default:
        return BlankLevel_Digital;
    }
}

nn::edid::SyncTypeSet nn::edid::detail::GetSyncTypes(std::uint8_t value) NN_NOEXCEPT
{
    nn::edid::SyncTypeSet sync;
    sync.Reset();

    // checking values of particular bits here
    sync.Set<SyncType::SeparateSync>((value & SeparateSyncMask) == 8);
    sync.Set<SyncType::CompositeSync>((value & CompositeSyncMask) == 4);
    sync.Set<SyncType::SyncOnGreen>((value & SyncOnGreenMask) == 2);

    return sync;
}

bool nn::edid::detail::HasVsyncSerrations(std::uint8_t value) NN_NOEXCEPT
{
    // analog/digital bit must be clear
    return (value & SerrationsMask) == 1;
}

nn::edid::ColorDepth nn::edid::detail::GetColorDepth(std::uint8_t value) NN_NOEXCEPT
{
    // analog/digital bit must be set
    switch( value & ColorDepthMask )
    {
    case 0x90:
        return ColorDepth_6Bits;
    case 0xA0:
        return ColorDepth_8Bits;
    case 0xB0:
        return ColorDepth_10Bits;
    case 0xC0:
        return ColorDepth_12Bits;
    case 0xD0:
        return ColorDepth_14Bits;
    case 0xE0:
        return ColorDepth_16Bits;
    default:
        return ColorDepth_Undefined;
    }
}

nn::edid::ColorType nn::edid::detail::GetColorSupport(VideoInterface interfaceType, std::uint8_t value) NN_NOEXCEPT
{
    if( interfaceType != VideoInterface_Analog )
    {
        return ColorType_Undefined;
    }

    switch( value & ColorSupportMask )
    {
    case 0x00:
        return ColorType_Monochrome;
    case 0x08:
        return ColorType_RgbColor;
    case 0x10:
        return ColorType_NonRgbColor;
    default:
        return ColorType_Undefined;
    }
}

bool nn::edid::detail::DisplayDataBlockVisitor(const std::uint8_t* pBlock, size_t size, cea861::BlockTag tag, void* pUserData) NN_NOEXCEPT
{
    DisplayInfo* pDisplay = static_cast<DisplayInfo*>(pUserData);

    switch( tag )
    {
    case cea861::BlockTag::VideoCapability:
        if( size > 2 )
        {
            pDisplay->isYccQuantizationSelectable = (pBlock[2] & YccQuantizationMask) != 0;
            pDisplay->isRgbQuantizationSelectable = (pBlock[2] & RgbQuantizationMask) != 0;
        }
        break;
    case cea861::BlockTag::VendorSpecific:
        if( size > 8 )
        {
            std::uint8_t flags = pBlock[0x08];

            pDisplay->contentTypes.Set<ContentType::Graphics>((flags & GraphicsTypeMask) != 0);
            pDisplay->contentTypes.Set<ContentType::Photo>((flags & PhotoTypeMask) != 0);
            pDisplay->contentTypes.Set<ContentType::Cinema>((flags & CinemaTypeMask) != 0);
            pDisplay->contentTypes.Set<ContentType::Game>((flags & GameTypeMask) != 0);
        }
        break;
    default:
        // Do nothing -- fixes warning for clang
        break;
    }

    return true;
}

bool nn::edid::detail::ExtensionBlockVisitor(const std::uint8_t* pBlock, size_t size, void* pUserData) NN_NOEXCEPT
{
    cea861::VisitDataBlocks(pBlock, size, DisplayDataBlockVisitor, pUserData);

    // TODO: break out early if we found everything?
    return true;
}

bool nn::edid::detail::IsStandbyModeSupported(std::uint8_t value) NN_NOEXCEPT
{
    return (value & StandbyModeMask) != 0;
}

bool nn::edid::detail::IsSuspendModeSupported(std::uint8_t value) NN_NOEXCEPT
{
    return (value & SuspendModeMask) != 0;
}

bool nn::edid::detail::IsVeryLowPowerSupported(std::uint8_t value) NN_NOEXCEPT
{
    return (value & VeryLowPowerMask) != 0;
}

bool nn::edid::detail::IsSrgbDefaultColorSpace(std::uint8_t value) NN_NOEXCEPT
{
    return (value & SrgbColorMask) != 0;
}

bool nn::edid::detail::IsContinuousFrequency(std::uint8_t value) NN_NOEXCEPT
{
    return (value & ContinuousFrequencyMask) != 0;
}

bool nn::edid::detail::PhysicalAddressVisitor(const std::uint8_t* pBlock, size_t size, void* pUserData) NN_NOEXCEPT
{
    PhysicalAddressData* pData = static_cast<PhysicalAddressData*>(pUserData);

    cea861::VisitDataBlocks(pBlock, size, PhysicalAddressDataBlockVisitor, pUserData);

    return !pData->isFound;
}

bool nn::edid::detail::PhysicalAddressDataBlockVisitor(const std::uint8_t* pBlock, size_t size, cea861::BlockTag tag, void* pUserData) NN_NOEXCEPT
{
    PhysicalAddressData* pData = static_cast<PhysicalAddressData*>(pUserData);

    switch( tag )
    {
    case cea861::BlockTag::VendorSpecific:
        // See Table 8-16 in HDMI 1.4b spec
        if( size > 5 )
        {
            pData->address = (pBlock[4] << 8) | pBlock[5];
            pData->isFound = true;
        }
        break;
    default:
        break;
    }

    return !pData->isFound;
}
