﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <cstdlib>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/diag/diag_AbortTypes.h>
#include <nn/diag/diag_LogTypes.h>

namespace nn { namespace diag { namespace detail {

namespace {

// Release ビルドで NN_SDK_LOG() が消えると ToString() は使用されない
#ifndef NN_SDK_BUILD_RELEASE
const char* ToString(
    AbortReason abortReason) NN_NOEXCEPT
{
    switch( abortReason )
    {
    case AbortReason_SdkAssert:
        return "SDK Assertion Failure";
    case AbortReason_SdkRequires:
        return "Precondition not met";
    case AbortReason_UserAssert:
        return "Assertion Failure";
    case AbortReason_UnexpectedDefault:
        return "Unexpected Default";
    case AbortReason_Abort:
    default:
        return "Abort";
    }
}
#endif

} // anonymous

void PrintAbortInfo(
    const AbortInfo &abortInfo) NN_NOEXCEPT
{
#ifdef NN_SDK_BUILD_RELEASE
    // Release ビルドで NN_SDK_LOG() が消えると abortInfo への参照がなくなる
    NN_UNUSED(abortInfo);
#endif

    NN_SDK_LOG(
        "%s: \'%s\' at '%s':%d in %s\n",
        ToString(abortInfo.abortReason),
        abortInfo.condition,
        abortInfo.fileName,
        abortInfo.lineNumber,
        abortInfo.functionName);
    NN_SDK_VLOG(abortInfo.message->format, *abortInfo.message->args);
    NN_SDK_LOG("\n");
}

NN_NORETURN void AbortImpl(
    const char* condition,
    const char* functionName,
    const char* fileName,
    int lineNumber,
    const char* format,
    ...) NN_NOEXCEPT
{
    std::va_list list;
    va_start(list, format);

    const nn::diag::LogMessage message = {format, &list};
    const AbortInfo abortInfo = {
        AbortReason_Abort,
        &message,
        condition,
        functionName,
        fileName,
        lineNumber
    };
    PrintAbortInfo(abortInfo);

    va_end(list);
    std::abort();
}

NN_NORETURN void AbortImpl(
    const char* condition,
    const char* functionName,
    const char* fileName,
    int lineNumber) NN_NOEXCEPT
{
    detail::AbortImpl(condition, functionName, fileName, lineNumber, "");
}

}}} // nn::diag::detail
