﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

namespace nn { namespace crypto { namespace detail {

/*
 * 各種 Update 処理のスケルトンです。
 * テンプレート引数 T で渡される各クラスに適切な関数が定義されていることを期待しています。
 * テンプレート引数 U は ProcessBlocks の実装をスイッチするために利用されます。
 */

// 入力のみの処理のテンプレート
template <typename U, typename T>
void UpdateImpl(T* self, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    const uint8_t* pSrc8 = static_cast<const uint8_t*>(pSrc);
    const size_t BlockSize = self->GetBlockSize();
    size_t remaining = srcSize;

    /* バッファされているデータがあった場合の処理 */
    if (self->GetBufferedDataSize() > 0)
    {
        size_t partialSize = std::min(BlockSize - self->GetBufferedDataSize(), remaining);

        self->ProcessPartialData(pSrc8, partialSize);
        pSrc8          += partialSize;
        remaining      -= partialSize;
    }

    /* ブロックサイズ以上の残りに対する処理。ブロックサイズ単位で入出力長が同じであることを仮定している */
    if (remaining >= BlockSize)
    {
        int numBlocks = static_cast<int>(remaining / BlockSize);

        self->template ProcessBlocks<U>(pSrc8, numBlocks);
        pSrc8          += numBlocks * BlockSize;
        remaining      -= numBlocks * BlockSize;
    }

    /* ブロックサイズ以下の端数が出た場合の処理 */
    if (remaining > 0)
    {
        self->ProcessRemainingData(pSrc8, remaining);
    }
}

// 入出力がある処理のテンプレート
template <typename U, typename T>
size_t UpdateImpl(T* self, void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    NN_UNUSED(dstSize);

    const size_t BlockSize = self->GetBlockSize();
    const uint8_t* pSrc8 = static_cast<const uint8_t*>(pSrc);
    uint8_t* pDst8 = static_cast<uint8_t*>(pDst);
    size_t remaining = srcSize;
    size_t processed = 0;
    size_t processedTotal = 0;

    /* バッファされているデータがあった場合の処理 */
    if (self->GetBufferedDataSize() > 0)
    {
        size_t partialSize = std::min(BlockSize - self->GetBufferedDataSize(), remaining);

        processed = self->ProcessPartialData(pDst8, pSrc8, partialSize);
        pDst8          += processed;
        processedTotal += processed;
        pSrc8          += partialSize;
        remaining      -= partialSize;
    }

    /* ブロックサイズ以上の残りに対する処理 */
    if (remaining >= BlockSize)
    {
        int numBlocks = static_cast<int>(remaining / BlockSize);

        processed = self->template ProcessBlocks<U>(pDst8, pSrc8, numBlocks);
        pDst8          += processed;
        processedTotal += processed;
        pSrc8          += numBlocks * BlockSize;
        remaining      -= numBlocks * BlockSize;
    }

    /* ブロックサイズ以下の端数が出た場合の処理 */
    if (remaining > 0)
    {
        processed = self->ProcessRemainingData(pDst8, pSrc8, remaining);
        processedTotal += processed;
    }

    return processedTotal;
}

}}} // nn::crypto::detail
