﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Config.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/detail/crypto_Sha256Impl.h>
#include <nn/crypto/detail/crypto_Clear.h>

#include "crypto_Util.h"

namespace nn { namespace crypto { namespace detail {

namespace
{
    /* 仕様書に定義されている定数値テーブル */
    const Bit32 ConstantTable[64] =
    {
        0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5,
        0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
        0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3,
        0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
        0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc,
        0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
        0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7,
        0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
        0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13,
        0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
        0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3,
        0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
        0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5,
        0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
        0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208,
        0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
    };

    /* Shr 関数 */
    inline Bit32 ShiftRight(Bit32 value, int shift) NN_NOEXCEPT
    {
        return (value >> shift);
    }

    /* Ch 関数 */
    inline Bit32 Choose(Bit32 x, Bit32 y, Bit32 z) NN_NOEXCEPT
    {
        return (((x) & (y)) ^ ((~x) & (z)));
    }

    /* Maj 関数 */
    inline Bit32 Majority(Bit32 x, Bit32 y, Bit32 z) NN_NOEXCEPT
    {
        return (((x) & (y)) ^ ((x) & (z)) ^ ((y) & (z)));
    }

    inline Bit32 LargeSigma0(Bit32 x) NN_NOEXCEPT
    {
        return (RotateRight(x, 2) ^ RotateRight(x, 13) ^ RotateRight(x, 22));
    }

    inline Bit32 LargeSigma1(Bit32 x) NN_NOEXCEPT
    {
        return (RotateRight(x, 6) ^ RotateRight(x, 11) ^ RotateRight(x, 25));
    }

    inline Bit32 SmallSigma0(Bit32 x) NN_NOEXCEPT
    {
        return (RotateRight(x, 7) ^ RotateRight(x, 18) ^ ShiftRight(x, 3));
    }

    inline Bit32 SmallSigma1(Bit32 x) NN_NOEXCEPT
    {
        return (RotateRight(x, 17) ^ RotateRight(x, 19) ^ ShiftRight(x, 10));
    }

}   // anonymous namespace


Sha256Impl::~Sha256Impl() NN_NOEXCEPT
{
    ClearMemory(this, sizeof(*this));
}

void Sha256Impl::Initialize() NN_NOEXCEPT
{
    m_InputBitCount = 0;
    m_BufferedByte = 0;
    m_IntermediateHash[0] = 0x6a09e667;
    m_IntermediateHash[1] = 0xbb67ae85;
    m_IntermediateHash[2] = 0x3c6ef372;
    m_IntermediateHash[3] = 0xa54ff53a;
    m_IntermediateHash[4] = 0x510e527f;
    m_IntermediateHash[5] = 0x9b05688c;
    m_IntermediateHash[6] = 0x1f83d9ab;
    m_IntermediateHash[7] = 0x5be0cd19;

    m_State = State_Initialized;
}

void Sha256Impl::Update(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_State == State_Initialized, "Invalid state. Please restart from Initialize().");

    /* 実際に処理される分だけデータサイズをビットサイズで加算していく */
    m_InputBitCount += 8 * ((m_BufferedByte + dataSize) / BlockSize) * BlockSize;

    const Bit8*  pData8 = static_cast<const Bit8*>(pData);
    size_t remaining = dataSize;

    /* 前の処理の残りがあったら1ブロックに到達するかデータが無くなるまで埋める */
    if (m_BufferedByte > 0)
    {
        size_t fillSize = std::min(BlockSize - m_BufferedByte, remaining);

        std::memcpy(m_TemporalBlockBuffer + m_BufferedByte, pData8, fillSize);
        pData8 += fillSize;
        remaining -= fillSize;
        m_BufferedByte += fillSize;
        if (m_BufferedByte == BlockSize)
        {
            ProcessBlock(m_TemporalBlockBuffer);
            m_BufferedByte = 0;
        }
    }

    /* ブロックサイズ以上の残りがある場合はブロックサイズごとに処理 */
    while (remaining >= BlockSize)
    {
        ProcessBlock(pData8);
        pData8 += BlockSize;
        remaining -= BlockSize;
    }

    /* ブロックサイズ以下の端数は次の処理のために保存しておく */
    if (remaining > 0)
    {
        m_BufferedByte = remaining;
        std::memcpy(m_TemporalBlockBuffer, pData8, remaining);
    }
}

void Sha256Impl::GetHash(void* pHash, size_t hashSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((m_State == State_Initialized) || (m_State == State_Done), "Invalid state. Please restart from Initialize().");
    NN_SDK_REQUIRES(hashSize >= HashSize, "It requires %d bytes buffer", HashSize);
    NN_UNUSED(hashSize);

    if (m_State == State_Initialized)
    {
        ProcessLastBlock();
        m_State = State_Done;
    }

#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    CopyDataWithSwappingEndianBy32Bit(pHash, m_IntermediateHash, HashSize);
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
    std::memcpy(pHash, m_IntermediateHash, HashSize);
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif
}

void Sha256Impl::ProcessBlock(const void* pData) NN_NOEXCEPT
{
    Bit32  a = m_IntermediateHash[0];
    Bit32  b = m_IntermediateHash[1];
    Bit32  c = m_IntermediateHash[2];
    Bit32  d = m_IntermediateHash[3];
    Bit32  e = m_IntermediateHash[4];
    Bit32  f = m_IntermediateHash[5];
    Bit32  g = m_IntermediateHash[6];
    Bit32  h = m_IntermediateHash[7];
    Bit32  w[64];
    Bit32  tmp1;
    Bit32  tmp2;
    Bit32* prev;
    int    t;

    /* 先頭の 16 word には BlockSize 分の入力データを格納する */
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    CopyDataWithSwappingEndianBy32Bit(w, pData, BlockSize);
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
    std::memcpy(w, pData, BlockSize);
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif

    for (t = 16; t < 64; t++)
    {
        prev = &w[t - 16];
        w[t] = prev[ 0]
             + SmallSigma0(prev[ 1])
             + prev[ 9]
             + SmallSigma1(prev[14]);
    }

    for (t = 0; t < 64; t++)
    {
        tmp1 = h
             + LargeSigma1(e)
             + Choose(e, f, g)
             + ConstantTable[t]
             + w[t];

        tmp2 = LargeSigma0(a)
             + Majority(a, b, c);

        h    = g;
        g    = f;
        f    = e;
        e    = d + tmp1;
        d    = c;
        c    = b;
        b    = a;
        a    = tmp1 + tmp2;
    }

    m_IntermediateHash[0] += a;
    m_IntermediateHash[1] += b;
    m_IntermediateHash[2] += c;
    m_IntermediateHash[3] += d;
    m_IntermediateHash[4] += e;
    m_IntermediateHash[5] += f;
    m_IntermediateHash[6] += g;
    m_IntermediateHash[7] += h;
}

void Sha256Impl::ProcessLastBlock() NN_NOEXCEPT
{
    const int BlockSizeWithoutSizeField = BlockSize - sizeof(Bit64);

    /* 最後にバッファされているデータ分のデータサイズを加算 */
    m_InputBitCount += 8 * m_BufferedByte;

    /* パディングの先頭を示す 0x80 を代入 */
    m_TemporalBlockBuffer[m_BufferedByte] = 0x80;
    m_BufferedByte++;

    /* 現在計算中のブロックにサイズを埋め込む余裕があるかないかで処理が変わる */
    if (m_BufferedByte <= BlockSizeWithoutSizeField)
    {
        /* そのままサイズを格納する領域の手前までパディング */
        std::memset(m_TemporalBlockBuffer + m_BufferedByte, 0x00, BlockSizeWithoutSizeField - m_BufferedByte);
    }
    else
    {
        /* このブロックは末尾までパディングしてハッシュ計算を行う */
        std::memset(m_TemporalBlockBuffer + m_BufferedByte, 0x00, BlockSize - m_BufferedByte);
        ProcessBlock(m_TemporalBlockBuffer);

        /* 次のブロックをサイズを格納する領域の手前までパディング */
        std::memset(m_TemporalBlockBuffer, 0x00, BlockSizeWithoutSizeField);
    }

    /* 最後の 8 バイトにメッセージの長さを入れてハッシュを計算 */
    Bit64 inputBitCount = Convert64BitToBigEndian(m_InputBitCount);
    std::memcpy(m_TemporalBlockBuffer + BlockSizeWithoutSizeField, &inputBitCount, sizeof(Bit64));

    ProcessBlock(m_TemporalBlockBuffer);
}

void Sha256Impl::InitializeWithContext(const Sha256Context* pContext) NN_NOEXCEPT
{
    std::memcpy(m_IntermediateHash, pContext->_intermediateHash, sizeof(m_IntermediateHash));
    m_InputBitCount = pContext->_inputBitCount;

    m_BufferedByte = 0;
    m_State = State_Initialized;
}

size_t Sha256Impl::GetContext(Sha256Context* pContext) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_State == State_Initialized);

    std::memcpy(pContext->_intermediateHash, m_IntermediateHash, sizeof(m_IntermediateHash));
    pContext->_inputBitCount = m_InputBitCount;

    return m_BufferedByte;
}

}}}
