﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Config.h>
#include <nn/crypto/detail/crypto_Sha1Impl.h>
#include <nn/crypto/detail/crypto_Clear.h>

#include "crypto_Util.h"

namespace nn { namespace crypto { namespace detail {

namespace
{
    /* 各ラウンドの処理で使用する定数 */
    const uint32_t RoundConstants[4] =
    {
        0x5a827999,
        0x6ed9eba1,
        0x8f1bbcdc,
        0xca62c1d6
    };

    /* Ch 関数 */
    inline Bit32 Choose(Bit32 x, Bit32 y, Bit32 z) NN_NOEXCEPT
    {
        return (((x) & (y)) ^ ((~x) & (z)));
    }

    /* Maj 関数 */
    inline Bit32 Majority(Bit32 x, Bit32 y, Bit32 z) NN_NOEXCEPT
    {
        return (((x) & (y)) ^ ((x) & (z)) ^ ((y) & (z)));
    }

    /* Parity 関数 */
    inline Bit32 Parity(Bit32 x, Bit32 y, Bit32 z) NN_NOEXCEPT
    {
        return ((x) ^ (y) ^ (z));
    }

}   // anonymous namespace


Sha1Impl::~Sha1Impl() NN_NOEXCEPT
{
    ClearMemory(this, sizeof(*this));
}

void Sha1Impl::Initialize() NN_NOEXCEPT
{
    m_InputBitCount = 0;
    m_BufferedByte = 0;

    m_IntermediateHash[0] = 0x67452301;
    m_IntermediateHash[1] = 0xefcdab89;
    m_IntermediateHash[2] = 0x98badcfe;
    m_IntermediateHash[3] = 0x10325476;
    m_IntermediateHash[4] = 0xc3d2e1f0;

    m_State = State_Initialized;
}

void Sha1Impl::Update(const void* pData, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_State == State_Initialized, "Invalid state. Please restart from Initialize().");

    /* 実際に処理される分だけデータサイズをビットサイズで加算していく */
    m_InputBitCount += 8 * ((m_BufferedByte + dataSize) / BlockSize) * BlockSize;

    const Bit8*  pData8 = static_cast<const Bit8*>(pData);
    size_t remaining = dataSize;

    /* 前の処理の残りがあったら1ブロックに到達するかデータが無くなるまで埋める */
    if (m_BufferedByte > 0)
    {
        size_t fillSize = std::min(BlockSize - m_BufferedByte, remaining);

        std::memcpy(m_TemporalBlockBuffer + m_BufferedByte, pData8, fillSize);
        pData8 += fillSize;
        remaining -= fillSize;
        m_BufferedByte += fillSize;
        if (m_BufferedByte == BlockSize)
        {
            ProcessBlock(m_TemporalBlockBuffer);
            m_BufferedByte = 0;
        }
    }

    /* ブロックサイズ以上の残りがある場合はブロックサイズごとに処理 */
    while (remaining >= BlockSize)
    {
        ProcessBlock(pData8);
        pData8 += BlockSize;
        remaining -= BlockSize;
    }

    /* ブロックサイズ以下の端数は次の処理のために保存しておく */
    if (remaining > 0)
    {
        m_BufferedByte = remaining;
        std::memcpy(m_TemporalBlockBuffer, pData8, remaining);
    }
}

void Sha1Impl::GetHash(void* pHash, size_t hashSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((m_State == State_Initialized) || (m_State == State_Done), "Invalid state. Please restart from Initialize().");
    NN_SDK_REQUIRES(hashSize >= HashSize, "It requires %d bytes buffer", HashSize);
    NN_UNUSED(hashSize);

    if (m_State == State_Initialized)
    {
        ProcessLastBlock();
        m_State = State_Done;
    }

#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    CopyDataWithSwappingEndianBy32Bit(pHash, m_IntermediateHash, HashSize);
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
    std::memcpy(pHash, m_IntermediateHash, HashSize);
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif
}

void Sha1Impl::ProcessBlock(const void* pData) NN_NOEXCEPT
{
    Bit32  a = m_IntermediateHash[0];
    Bit32  b = m_IntermediateHash[1];
    Bit32  c = m_IntermediateHash[2];
    Bit32  d = m_IntermediateHash[3];
    Bit32  e = m_IntermediateHash[4];
    Bit32  w[80];
    Bit32  tmp;
    int    t;

    /* 先頭の 16 word には BlockSize 分の入力データを格納する */
#if defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    CopyDataWithSwappingEndianBy32Bit(w, pData, BlockSize);
#elif defined(NN_BUILD_CONFIG_ENDIAN_BIG)
    std::memcpy(w, pData, BlockSize);
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif

    /* 残りの 64 word はメッセージを拡張する */
    for (t = 16; t < 80; t++)
    {
        volatile Bit32* prev = &w[t - 16]; /* 意図しない最適化を防止する */
        w[t] = RotateLeft(prev[ 0] ^ prev[ 2] ^ prev[ 8] ^ prev[13], 1);
    }

    for (t = 0; t < 20; t++)
    {
        tmp = RotateLeft(a, 5) + Choose(b, c ,d) + e + w[t] + RoundConstants[0];
        e = d;
        d = c;
        c = RotateLeft(b, 30);
        b = a;
        a = tmp;
    }

    for (; t < 40; t++)
    {
        tmp = RotateLeft(a, 5) + Parity(b, c ,d) + e + w[t] + RoundConstants[1];
        e = d;
        d = c;
        c = RotateLeft(b, 30);
        b = a;
        a = tmp;
    }

    for (; t < 60; t++)
    {
        tmp = RotateLeft(a, 5) + Majority(b, c ,d) + e + w[t] + RoundConstants[2];
        e = d;
        d = c;
        c = RotateLeft(b, 30);
        b = a;
        a = tmp;
    }

    for (; t < 80; t++)
    {
        tmp = RotateLeft(a, 5) + Parity(b, c ,d) + e + w[t] + RoundConstants[3];
        e = d;
        d = c;
        c = RotateLeft(b, 30);
        b = a;
        a = tmp;
    }

    m_IntermediateHash[0] += a;
    m_IntermediateHash[1] += b;
    m_IntermediateHash[2] += c;
    m_IntermediateHash[3] += d;
    m_IntermediateHash[4] += e;
}

void Sha1Impl::ProcessLastBlock() NN_NOEXCEPT
{
    const int BlockSizeWithoutSizeField = BlockSize - sizeof(Bit64);

    /* 最後にバッファされているデータ分のデータサイズを加算 */
    m_InputBitCount += 8 * m_BufferedByte;

    /* パディングの先頭を示す 0x80 を代入 */
    m_TemporalBlockBuffer[m_BufferedByte] = 0x80;
    m_BufferedByte++;

    /* 現在計算中のブロックにサイズを埋め込む余裕があるかないかで処理が変わる */
    if (m_BufferedByte <= BlockSizeWithoutSizeField)
    {
        /* そのままサイズを格納する領域の手前までパディング */
        std::memset(m_TemporalBlockBuffer + m_BufferedByte, 0x00, BlockSizeWithoutSizeField - m_BufferedByte);
    }
    else
    {
        /* このブロックは末尾までパディングしてハッシュ計算を行う */
        std::memset(m_TemporalBlockBuffer + m_BufferedByte, 0x00, BlockSize - m_BufferedByte);
        ProcessBlock(m_TemporalBlockBuffer);

        /* 次のブロックをサイズを格納する領域の手前までパディング */
        std::memset(m_TemporalBlockBuffer, 0x00, BlockSizeWithoutSizeField);
    }

    /* 最後の 8 バイトにメッセージの長さを入れてハッシュを計算 */
    Bit64 inputBitCount = Convert64BitToBigEndian(m_InputBitCount);
    std::memcpy(m_TemporalBlockBuffer + BlockSizeWithoutSizeField, &inputBitCount, sizeof(Bit64));

    ProcessBlock(m_TemporalBlockBuffer);
}

}}}
