﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "crypto_BigNumMath.h"

namespace nn { namespace crypto { namespace detail {

enum
{
    ECC_P256_BIT_LENGTH = 256,
    ECC_P256_BIGINT_DIGITS = ((ECC_P256_BIT_LENGTH + BigNum::DigitBits - 1) / BigNum::DigitBits),
    ECC_P256_OCTET_LENGTH = ((ECC_P256_BIT_LENGTH + 7) / 8)
};

// a point on the curve. By convention, point at infinity is 0, 0
typedef struct p256_point
{
    BigNum::Digit x[ECC_P256_BIGINT_DIGITS];
    BigNum::Digit y[ECC_P256_BIGINT_DIGITS];
} p256_point;

typedef struct EccPrivateKey
{
    BigNum::Digit p[ECC_P256_BIGINT_DIGITS];
} EccPrivateKey;

/*
q = 256
FR = basis ind
c, b = elements of Fq
G = generator
n = order of G
h = curve order / n
*/

/* E := y^2 = x^3 - a4 x + a6 */
typedef struct
{
    BigNum::Digit a4[ECC_P256_BIGINT_DIGITS]; // corresponds to the opposite of the parameter in x
    BigNum::Digit a6[ECC_P256_BIGINT_DIGITS]; // corresponds to the constant parameter
} p256_curve;

typedef struct
{
    p256_curve par_curve;
    p256_point par_point;
    BigNum::Digit point_order[ECC_P256_BIGINT_DIGITS];
    BigNum::Digit cofactor[ECC_P256_BIGINT_DIGITS];
} p256_ec_parameter;

/* verify that a point is on our curve (this should be called on adversary input) */
int p256_validate(const p256_point* a, const p256_curve* c);
void p256_add(p256_point* res, const p256_point* a, const p256_point* b, const p256_curve* curv);
void p256_mul(p256_point* res, const BigNum::Digit* n, int n_digits, const p256_point* a,
              const p256_curve* curv);

extern const p256_ec_parameter p256_named_parameter;

}}} // namespace nn::crypto::detail
