﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/detail/crypto_BigNum.h>
#include <nn/crypto/detail/crypto_EccP256Key.h>
#include "crypto_EccP256Algorithms.h"

namespace nn { namespace crypto { namespace detail {

inline const struct p256_point* EccP256Key::GetPublicKey() const
{
    NN_SDK_ASSERT(m_Model.m_HasPublicKey);
    return reinterpret_cast<const p256_point*>(&m_Model.m_PublicKey);
}

bool EccP256Key::SignHash(void* outSignature, size_t signatureSize,
                          const void* pHash, size_t hashSize,
                          const void* random, size_t randomSize) const NN_NOEXCEPT
{
    NN_UNUSED(signatureSize);
    NN_UNUSED(randomSize);
    NN_SDK_ASSERT(m_Model.m_HasPrivateKey);

    BigNum::Digit randomFe[Model::DigitsCount];
    DigitsFromOctetString( randomFe, Model::DigitsCount, random, randomSize );

    p256_ec_signature signature;
    CSL_error ret = alg_p256_ECDSA_sign(&signature,
                                        pHash, hashSize,
                                        &p256_named_parameter,
                                        m_Model.m_PrivateKey,
                                        randomFe, Model::DigitsCount);

    if (ret == CSL_OK)
    {
        OctetStringFromDigits(outSignature, SignatureSize / 2, signature.r, Model::DigitsCount);
        OctetStringFromDigits(static_cast<char*>(outSignature) + SignatureSize / 2,
                              SignatureSize / 2,
                              signature.s, Model::DigitsCount);
        return true;
    }
    return false;

}

bool EccP256Key::VerifyHash(const void* pSignature, size_t signatureSize,
                            const void* pHash, size_t hashSize ) const NN_NOEXCEPT
{
    NN_UNUSED(signatureSize);
    NN_SDK_ASSERT(m_Model.m_HasPublicKey);

    p256_ec_signature signature;
    DigitsFromOctetString(signature.r, ECC_P256_BIGINT_DIGITS, pSignature, SignatureSize / 2);
    DigitsFromOctetString(signature.s, ECC_P256_BIGINT_DIGITS,
                          static_cast<const char*>(pSignature) + SignatureSize / 2,
                          SignatureSize / 2);

    CSL_error ret = alg_p256_ECDSA_verify(&signature,
                                          pHash, hashSize,
                                          &p256_named_parameter,
                                          GetPublicKey() );

    return (ret == CSL_OK);
}

bool EccP256Key::GenerateSharedSecret( void* outSharedSecret, size_t sharedSecretSize,
                                       const EccP256Key& recipientPublicKey ) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Model.m_HasPrivateKey);
    NN_SDK_ASSERT(recipientPublicKey.m_Model.m_HasPublicKey);

    if (sharedSecretSize != SharedSecretSize)
    {
        return false;
    }

    BigNum::Digit shkey[ECC_P256_BIGINT_DIGITS];
    CSL_error error = alg_p256_generate_shared_key( shkey,
                                                    &(p256_named_parameter.par_point),
                                                    &(p256_named_parameter.par_curve),
                                                    recipientPublicKey.GetPublicKey(),
                                                    m_Model.m_PrivateKey );
    if (error != CSL_OK)
    {
        return false;
    }

    OctetStringFromDigits(outSharedSecret, sharedSecretSize, shkey, ECC_P256_BIGINT_DIGITS);
    return true;
}

}}} // namespace nn::crypto::detail
