﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <nn/crypto/detail/crypto_BigNum.h>
#include "crypto_EccP256Algorithms.h"

#define CSL_UNUSED(var) ((void)&var)

namespace nn { namespace crypto { namespace detail {

CSL_error
alg_p256_generate_public_key(p256_point* base_point, p256_curve* E,
                             BigNum::Digit* my_private, p256_point* my_public)
{
    p256_mul(my_public, my_private, ECC_P256_BIGINT_DIGITS, base_point, E);
    if (p256_validate(my_public, E))
    {
        return CSL_OK;
    }
    else
    {
        return CSL_BAD_KEY;
    }
}

/*
 * Generate ECDH shared key from private and public keys
 */
CSL_error
alg_p256_generate_shared_key(BigNum::Digit* shared_secret,
                             const p256_point* base_point, const p256_curve* E,
                             const p256_point* recipient_public, const BigNum::Digit* my_private )
{
    p256_point tmp;
    CSL_UNUSED(base_point);
    if (!p256_validate(recipient_public, &(p256_named_parameter.par_curve)))
    {
        return CSL_BAD_KEY;
    }

    p256_mul(&tmp, my_private, ECC_P256_BIGINT_DIGITS, recipient_public, E);
    BigNum::Copy(shared_secret, tmp.x, ECC_P256_BIGINT_DIGITS);

    return CSL_OK;
}

/*
 * generate random key pair, needed for ECDSA
 * inputs random number: replace with one from
 * hardware
 * there should be at least 256 + 64 random bits to
 * ensure generation is uniformly distributed!

Suite B Implementer’s Guide to FIPS 186-3
Appendix A.2.1 (p20) : ECC Per-Message Secret Number Generation Using Extra Random Bits

 */

CSL_error
alg_p256_generate_private_key(const BigNum::Digit* rand_input, int rand_digits,
                              const p256_ec_parameter* public_curve, BigNum::Digit* outPrivate)
{

    BigNum::Digit one[ECC_P256_BIGINT_DIGITS] = {
        1, 0, 0, 0, 0, 0, 0, 0
    };
    BigNum::Digit order_minus_one[ECC_P256_BIGINT_DIGITS];
    int point_order_digits;

    if (rand_digits > 2 * ECC_P256_BIGINT_DIGITS)
    {
        rand_digits = 2 * ECC_P256_BIGINT_DIGITS;
    }

    if (BigNum::IsZero(rand_input, rand_digits))
    {
        return(CSL_VERIFY_ERROR);
    }
    rand_digits = BigNum::GetDigits(rand_input, rand_digits);

    BigNum::Sub(order_minus_one, public_curve->point_order, one, ECC_P256_BIGINT_DIGITS);
    point_order_digits = BigNum::GetDigits(order_minus_one, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(outPrivate, ECC_P256_BIGINT_DIGITS);
    BigNum::Mod(outPrivate, rand_input, rand_digits, order_minus_one, point_order_digits);
    BigNum::Add(outPrivate, outPrivate, one, ECC_P256_BIGINT_DIGITS);
    // outPrivate (d or k) : = (c mod (n − 1)) + 1
    return CSL_OK;
}

CSL_error
loc_p256_key_gen_primitive(p256_ec_keypair* outKeyPair, const p256_ec_parameter* base,
                           const BigNum::Digit* random_input, int random_digits)
{
    // CAVS test vectors and rfc6979 use random directly, as in B-233
    // private : k mod n
    /*
    CSL_error error;
    if ((error =
             alg_p256_generate_private_key(random_input, random_digits, base,
                                           outKeyPair->private_key)) != CSL_OK)
    {
        return error;
    }
    */
    BigNum::Mod( outKeyPair->private_key, random_input, random_digits, base->point_order,
                ECC_P256_BIGINT_DIGITS);

    // R = kG
    p256_mul(&outKeyPair->public_key, outKeyPair->private_key, ECC_P256_BIGINT_DIGITS,
             &base->par_point,
             &base->par_curve);
    return CSL_OK;

}

CSL_error
alg_p256_generate_shared_key_pre(p256_point* base_point, p256_curve* E,
                                 p256_point* recipient_public, BigNum::Digit* my_private,
                                 BigNum::Digit* shared_secret)
{
    p256_point tmp;
    CSL_UNUSED(base_point);
    if (!p256_validate(recipient_public, &(p256_named_parameter.par_curve)))
    {
        return CSL_BAD_KEY;
    }

    p256_mul(&tmp, my_private, ECC_P256_BIGINT_DIGITS, recipient_public, E);
    BigNum::Copy(shared_secret, tmp.x, ECC_P256_BIGINT_DIGITS);

    return CSL_OK;
}

// init ECC global state;

void
alg_init_p256_ECDSA(p256_ec_parameter* base)
{
    memcpy(base, &p256_named_parameter, sizeof(p256_ec_parameter));
}

static int
loc_int_po_range(const BigNum::Digit* i, const BigNum::Digit* po, int ndigits)
{
    if (BigNum::IsZero(i, ndigits))
    {
        return(0);
    }
    if (BigNum::Compare(i, po, ndigits) >= 0)
    {
        return(0);
    }
    return(1);
}
/*
// Appendix B.2 Conversion of a Bit String to an Integer
static void DigitsFromBitString( BigNum::Digit* outDigits, int digitsCount, const void* inOctets, int octetsLen )
{
    for( ; digitsCount-->0 && octetsLen>0; )
    {
        BigNum::Digit outDigit = 0;
        BigNum::Digit outMask = ((BigNum::Digit)1)<<(BigNum::DigitBits-1);
        for( int j=0; j<sizeof(BigNum::Digit) && octetsLen>0; ++j )
        {
            uint8_t inByte = ((const uint8_t*)inOctets)[--octetsLen];
            for( int bit=0; bit<8; ++bit )
            {
                if( (inByte & (1<<bit)) != 0 )
                {
                    outDigit |= outMask;
                }
                outMask >>= 1;
            }
        }
        *outDigits++ = outDigit;
    }
    while( digitsCount-->0 )
    {
        *outDigits++ = 0;
    }
}
*/
/* This is algorithm for ECDSA, enter with message pointer, size,
 * private key, and get signature: two points
 * there must be enough random bits (at least 256 + 64)
 */
CSL_error
alg_p256_ECDSA_sign(p256_ec_signature* outSignature,
                    const void* messageDigest, size_t messageDigestSize,
                    const p256_ec_parameter* public_curve, const BigNum::Digit* private_key,
                    const BigNum::Digit* rand_input, size_t rand_digits)
{
    BigNum::Digit hash_value[2 * ECC_P256_BIGINT_DIGITS];
    BigNum::Digit k_value[ECC_P256_BIGINT_DIGITS];
    BigNum::Digit temp[ECC_P256_BIGINT_DIGITS * 2];
    BigNum::Digit u_value[ECC_P256_BIGINT_DIGITS];
    int x_value_digits, point_order_digits, temp_digits;
    p256_ec_keypair random_key;
    CSL_error err;

    BigNum::SetZero(k_value, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(temp, ECC_P256_BIGINT_DIGITS * 2);
    BigNum::SetZero(u_value, ECC_P256_BIGINT_DIGITS);

    BigNum::SetZero(hash_value, ECC_P256_BIGINT_DIGITS * 2);
    /*
        extern CSL_error
        loc_hash_to_integer(const char *messagedigest, int length, BigNum::Digit *hash_value, int num_bits);
        loc_hash_to_integer((const char*)messageDigest, messageDigestSize, hash_value, ECC_P256_OCTET_LENGTH*8);
    */
    if ( messageDigestSize > ECC_P256_OCTET_LENGTH ) // truncate e.g. SHA-384
    {
        messageDigestSize = ECC_P256_OCTET_LENGTH;
    }
    DigitsFromOctetString(hash_value, ECC_P256_BIGINT_DIGITS, messageDigest, messageDigestSize);

    // public: R = kG (xR, yR)    private: k
    err =
        loc_p256_key_gen_primitive(&random_key, public_curve, rand_input,
                                   static_cast<int>(rand_digits));
    if (err != CSL_OK)
    {
        return(err);
    }

    x_value_digits = BigNum::GetDigits(random_key.public_key.x, ECC_P256_BIGINT_DIGITS);
    if (BigNum::IsZero(random_key.public_key.x, x_value_digits))
    {
        return CSL_DIVIDE_BY_ZERO;
    }

    point_order_digits = BigNum::GetDigits(public_curve->point_order, ECC_P256_BIGINT_DIGITS);
    // r = xR mod n
    BigNum::Mod(outSignature->r, random_key.public_key.x, x_value_digits, public_curve->point_order,
               point_order_digits);
    /*
     * check if component calculated is zero. check for attack
     * described in ECDSA-johnstone, menezes, vanstone
     */
    if (BigNum::IsZero(outSignature->r, ECC_P256_BIGINT_DIGITS))
    {
        return CSL_DIVIDE_BY_ZERO;
    }

    /* hash + priv key *c_value */
    // temp = d * r
    BigNum::Mult(temp, private_key, outSignature->r, ECC_P256_BIGINT_DIGITS);
    // temp = e + d*r
    BigNum::Add(temp, hash_value, temp, 2 * ECC_P256_BIGINT_DIGITS);

    temp_digits = BigNum::GetDigits(temp, ECC_P256_BIGINT_DIGITS * 2);
    // k_value = (e + d*r) mod n
    BigNum::Mod(k_value, temp, temp_digits, public_curve->point_order, point_order_digits);

    // u_value = 1/k
    /* multiply by inv of random key mod order of base point */
    BigNum::ModInv(u_value, random_key.private_key, public_curve->point_order,
                   ECC_P256_BIGINT_DIGITS);

    // temp = 1/k * ((e + d*r) mod n)
    BigNum::Mult(temp, u_value, k_value, ECC_P256_BIGINT_DIGITS);
    temp_digits = BigNum::GetDigits(temp, ECC_P256_BIGINT_DIGITS * 2);
    // s = (1/k * ((e + d*r) mod n)) mod n
    BigNum::Mod(outSignature->s, temp, temp_digits, public_curve->point_order, point_order_digits);

    if (BigNum::IsZero(outSignature->s, ECC_P256_BIGINT_DIGITS))
    {
        return CSL_DIVIDE_BY_ZERO;
    }
    return CSL_OK;
}

/*
 * Call to verify ECDSA 1:verifies, 0: not verifies
 */
CSL_error alg_p256_ECDSA_verify(const p256_ec_signature* signature,
                                const void* messageDigest, size_t messageDigestSize,
                                const p256_ec_parameter* public_curve,
                                const p256_point* signer_point)
{

    BigNum::Digit hash_value[ECC_P256_BIGINT_DIGITS];
    BigNum::Digit d_value[ECC_P256_BIGINT_DIGITS];
    BigNum::Digit temp[2 * ECC_P256_BIGINT_DIGITS];
    BigNum::Digit h1[ECC_P256_BIGINT_DIGITS], h2[ECC_P256_BIGINT_DIGITS];
    BigNum::Digit check_value[ECC_P256_BIGINT_DIGITS], point_order[ECC_P256_BIGINT_DIGITS];

    p256_point Temp1, Temp2;
    int temp_digits, point_order_digits;

    BigNum::SetZero(hash_value, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(d_value, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(temp, 2 * ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(h1, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(h2, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(check_value, ECC_P256_BIGINT_DIGITS);
    BigNum::SetZero(point_order, ECC_P256_BIGINT_DIGITS);

    if (!p256_validate(signer_point, &(p256_named_parameter.par_curve)))
    {
        return CSL_BAD_KEY;
    }

    // check r/s values in range [1,point_order-1];
    /* compute inv of second value */

    if ( !loc_int_po_range(signature->r, public_curve->point_order, ECC_P256_BIGINT_DIGITS))
    {
        return(CSL_VERIFY_ERROR);
    }
    if ( !loc_int_po_range(signature->s, public_curve->point_order, ECC_P256_BIGINT_DIGITS))
    {
        return(CSL_VERIFY_ERROR);
    }
    BigNum::ModInv(d_value, signature->s, public_curve->point_order, ECC_P256_BIGINT_DIGITS);

    /* generate hash */
    if ( messageDigestSize > ECC_P256_OCTET_LENGTH ) // truncate e.g. SHA-384
    {
        messageDigestSize = ECC_P256_OCTET_LENGTH;
    }
    DigitsFromOctetString(hash_value, ECC_P256_BIGINT_DIGITS, messageDigest, messageDigestSize);

    /* compute h1, h2*/
    BigNum::Mult(temp, hash_value, d_value, ECC_P256_BIGINT_DIGITS);
    temp_digits = BigNum::GetDigits(temp, ECC_P256_BIGINT_DIGITS * 2);
    point_order_digits = BigNum::GetDigits(public_curve->point_order, ECC_P256_BIGINT_DIGITS);
    BigNum::Mod(h1, temp, temp_digits, public_curve->point_order, point_order_digits);

    BigNum::Mult(temp, d_value, signature->r, ECC_P256_BIGINT_DIGITS);

    temp_digits = BigNum::GetDigits(temp, ECC_P256_BIGINT_DIGITS * 2);
    BigNum::Mod(h2, temp, temp_digits, public_curve->point_order, point_order_digits);

    p256_mul(&Temp1, h1, ECC_P256_BIGINT_DIGITS, &public_curve->par_point,
             &public_curve->par_curve);
    p256_mul(&Temp2, h2, ECC_P256_BIGINT_DIGITS, signer_point, &public_curve->par_curve);
    p256_add(&Temp1, &Temp1, &Temp2, &public_curve->par_curve);

    /* if point is zero, return false */
    if ( BigNum::IsZero(Temp1.x, ECC_P256_BIGINT_DIGITS)
         && BigNum::IsZero(Temp1.y, ECC_P256_BIGINT_DIGITS))
    {
        return CSL_DIVIDE_BY_ZERO;
    }

    if (BigNum::Compare(Temp1.x, signature->r, ECC_P256_BIGINT_DIGITS) == 0)
    {
        return CSL_OK;
    }
    return CSL_VERIFY_ERROR;
}

}}} // namespace nn::crypto::detail
