﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Config.h>
#include <nn/crypto/detail/crypto_AesImpl.h>
#include <nn/crypto/detail/crypto_Clear.h>

namespace nn { namespace crypto { namespace detail {

namespace
{
    const int BlockWords = AesImpl<16>::BlockSize / sizeof(Bit32);

    /* SubBytes 処理の置換表 */
    const Bit8 SubBytesTable[256] =
    {
        0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
        0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
        0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
        0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
        0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
        0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
        0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
        0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
        0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
        0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
        0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
        0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
        0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
        0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
        0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
        0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16,
    };

    /* InvSubBytes 処理の置換表 */
    const Bit8 InvSubBytesTable[256] =
    {
        0x52 ,0x09 ,0x6a ,0xd5 ,0x30 ,0x36 ,0xa5 ,0x38 ,0xbf ,0x40 ,0xa3 ,0x9e ,0x81 ,0xf3 ,0xd7 ,0xfb,
        0x7c ,0xe3 ,0x39 ,0x82 ,0x9b ,0x2f ,0xff ,0x87 ,0x34 ,0x8e ,0x43 ,0x44 ,0xc4 ,0xde ,0xe9 ,0xcb,
        0x54 ,0x7b ,0x94 ,0x32 ,0xa6 ,0xc2 ,0x23 ,0x3d ,0xee ,0x4c ,0x95 ,0x0b ,0x42 ,0xfa ,0xc3 ,0x4e,
        0x08 ,0x2e ,0xa1 ,0x66 ,0x28 ,0xd9 ,0x24 ,0xb2 ,0x76 ,0x5b ,0xa2 ,0x49 ,0x6d ,0x8b ,0xd1 ,0x25,
        0x72 ,0xf8 ,0xf6 ,0x64 ,0x86 ,0x68 ,0x98 ,0x16 ,0xd4 ,0xa4 ,0x5c ,0xcc ,0x5d ,0x65 ,0xb6 ,0x92,
        0x6c ,0x70 ,0x48 ,0x50 ,0xfd ,0xed ,0xb9 ,0xda ,0x5e ,0x15 ,0x46 ,0x57 ,0xa7 ,0x8d ,0x9d ,0x84,
        0x90 ,0xd8 ,0xab ,0x00 ,0x8c ,0xbc ,0xd3 ,0x0a ,0xf7 ,0xe4 ,0x58 ,0x05 ,0xb8 ,0xb3 ,0x45 ,0x06,
        0xd0 ,0x2c ,0x1e ,0x8f ,0xca ,0x3f ,0x0f ,0x02 ,0xc1 ,0xaf ,0xbd ,0x03 ,0x01 ,0x13 ,0x8a ,0x6b,
        0x3a ,0x91 ,0x11 ,0x41 ,0x4f ,0x67 ,0xdc ,0xea ,0x97 ,0xf2 ,0xcf ,0xce ,0xf0 ,0xb4 ,0xe6 ,0x73,
        0x96 ,0xac ,0x74 ,0x22 ,0xe7 ,0xad ,0x35 ,0x85 ,0xe2 ,0xf9 ,0x37 ,0xe8 ,0x1c ,0x75 ,0xdf ,0x6e,
        0x47 ,0xf1 ,0x1a ,0x71 ,0x1d ,0x29 ,0xc5 ,0x89 ,0x6f ,0xb7 ,0x62 ,0x0e ,0xaa ,0x18 ,0xbe ,0x1b,
        0xfc ,0x56 ,0x3e ,0x4b ,0xc6 ,0xd2 ,0x79 ,0x20 ,0x9a ,0xdb ,0xc0 ,0xfe ,0x78 ,0xcd ,0x5a ,0xf4,
        0x1f ,0xdd ,0xa8 ,0x33 ,0x88 ,0x07 ,0xc7 ,0x31 ,0xb1 ,0x12 ,0x10 ,0x59 ,0x27 ,0x80 ,0xec ,0x5f,
        0x60 ,0x51 ,0x7f ,0xa9 ,0x19 ,0xb5 ,0x4a ,0x0d ,0x2d ,0xe5 ,0x7a ,0x9f ,0x93 ,0xc9 ,0x9c ,0xef,
        0xa0 ,0xe0 ,0x3b ,0x4d ,0xae ,0x2a ,0xf5 ,0xb0 ,0xc8 ,0xeb ,0xbb ,0x3c ,0x83 ,0x53 ,0x99 ,0x61,
        0x17 ,0x2b ,0x04 ,0x7e ,0xba ,0x77 ,0xd6 ,0x26 ,0xe1 ,0x69 ,0x14 ,0x63 ,0x55 ,0x21 ,0x0c ,0x7d,
    };

    /* 鍵拡張で使用される定数 */
    const Bit8 RoundKeyRcon0[] =
    {
        0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80,
        0x1B, 0x36, 0x6C, 0xD8, 0xAB, 0x4D, 0x9A, 0x2F,
        0x5E, 0xBC, 0x63, 0xC6, 0x97, 0x35, 0x6A, 0xD4,
        0xB3, 0x7D, 0xFA, 0xEF, 0xC5, 0x91,
    };

#if defined(NN_BUILD_CONFIG_ENDIAN_BIG)
    // ビッグエンディアン
    const int aesWordByte0 = 24UL;
    const int aesWordByte1 = 16UL;
    const int aesWordByte2 = 8UL;
    const int aesWordByte3 = 0UL;

    const int MixShift = 8UL;

    const Bit32 EncryptTable[256] =
    {
        0x6363A5C6,0x7C7C84F8,0x777799EE,0x7B7B8DF6,0xF2F20DFF,0x6B6BBDD6,0x6F6FB1DE,0xC5C55491,
        0x30305060,0x01010302,0x6767A9CE,0x2B2B7D56,0xFEFE19E7,0xD7D762B5,0xABABE64D,0x76769AEC,
        0xCACA458F,0x82829D1F,0xC9C94089,0x7D7D87FA,0xFAFA15EF,0x5959EBB2,0x4747C98E,0xF0F00BFB,
        0xADADEC41,0xD4D467B3,0xA2A2FD5F,0xAFAFEA45,0x9C9CBF23,0xA4A4F753,0x727296E4,0xC0C05B9B,
        0xB7B7C275,0xFDFD1CE1,0x9393AE3D,0x26266A4C,0x36365A6C,0x3F3F417E,0xF7F702F5,0xCCCC4F83,
        0x34345C68,0xA5A5F451,0xE5E534D1,0xF1F108F9,0x717193E2,0xD8D873AB,0x31315362,0x15153F2A,
        0x04040C08,0xC7C75295,0x23236546,0xC3C35E9D,0x18182830,0x9696A137,0x05050F0A,0x9A9AB52F,
        0x0707090E,0x12123624,0x80809B1B,0xE2E23DDF,0xEBEB26CD,0x2727694E,0xB2B2CD7F,0x75759FEA,
        0x09091B12,0x83839E1D,0x2C2C7458,0x1A1A2E34,0x1B1B2D36,0x6E6EB2DC,0x5A5AEEB4,0xA0A0FB5B,
        0x5252F6A4,0x3B3B4D76,0xD6D661B7,0xB3B3CE7D,0x29297B52,0xE3E33EDD,0x2F2F715E,0x84849713,
        0x5353F5A6,0xD1D168B9,0x00000000,0xEDED2CC1,0x20206040,0xFCFC1FE3,0xB1B1C879,0x5B5BEDB6,
        0x6A6ABED4,0xCBCB468D,0xBEBED967,0x39394B72,0x4A4ADE94,0x4C4CD498,0x5858E8B0,0xCFCF4A85,
        0xD0D06BBB,0xEFEF2AC5,0xAAAAE54F,0xFBFB16ED,0x4343C586,0x4D4DD79A,0x33335566,0x85859411,
        0x4545CF8A,0xF9F910E9,0x02020604,0x7F7F81FE,0x5050F0A0,0x3C3C4478,0x9F9FBA25,0xA8A8E34B,
        0x5151F3A2,0xA3A3FE5D,0x4040C080,0x8F8F8A05,0x9292AD3F,0x9D9DBC21,0x38384870,0xF5F504F1,
        0xBCBCDF63,0xB6B6C177,0xDADA75AF,0x21216342,0x10103020,0xFFFF1AE5,0xF3F30EFD,0xD2D26DBF,
        0xCDCD4C81,0x0C0C1418,0x13133526,0xECEC2FC3,0x5F5FE1BE,0x9797A235,0x4444CC88,0x1717392E,
        0xC4C45793,0xA7A7F255,0x7E7E82FC,0x3D3D477A,0x6464ACC8,0x5D5DE7BA,0x19192B32,0x737395E6,
        0x6060A0C0,0x81819819,0x4F4FD19E,0xDCDC7FA3,0x22226644,0x2A2A7E54,0x9090AB3B,0x8888830B,
        0x4646CA8C,0xEEEE29C7,0xB8B8D36B,0x14143C28,0xDEDE79A7,0x5E5EE2BC,0x0B0B1D16,0xDBDB76AD,
        0xE0E03BDB,0x32325664,0x3A3A4E74,0x0A0A1E14,0x4949DB92,0x06060A0C,0x24246C48,0x5C5CE4B8,
        0xC2C25D9F,0xD3D36EBD,0xACACEF43,0x6262A6C4,0x9191A839,0x9595A431,0xE4E437D3,0x79798BF2,
        0xE7E732D5,0xC8C8438B,0x3737596E,0x6D6DB7DA,0x8D8D8C01,0xD5D564B1,0x4E4ED29C,0xA9A9E049,
        0x6C6CB4D8,0x5656FAAC,0xF4F407F3,0xEAEA25CF,0x6565AFCA,0x7A7A8EF4,0xAEAEE947,0x08081810,
        0xBABAD56F,0x787888F0,0x25256F4A,0x2E2E725C,0x1C1C2438,0xA6A6F157,0xB4B4C773,0xC6C65197,
        0xE8E823CB,0xDDDD7CA1,0x74749CE8,0x1F1F213E,0x4B4BDD96,0xBDBDDC61,0x8B8B860D,0x8A8A850F,
        0x707090E0,0x3E3E427C,0xB5B5C471,0x6666AACC,0x4848D890,0x03030506,0xF6F601F7,0x0E0E121C,
        0x6161A3C2,0x35355F6A,0x5757F9AE,0xB9B9D069,0x86869117,0xC1C15899,0x1D1D273A,0x9E9EB927,
        0xE1E138D9,0xF8F813EB,0x9898B32B,0x11113322,0x6969BBD2,0xD9D970A9,0x8E8E8907,0x9494A733,
        0x9B9BB62D,0x1E1E223C,0x87879215,0xE9E920C9,0xCECE4987,0x5555FFAA,0x28287850,0xDFDF7AA5,
        0x8C8C8F03,0xA1A1F859,0x89898009,0x0D0D171A,0xBFBFDA65,0xE6E631D7,0x4242C684,0x6868B8D0,
        0x4141C382,0x9999B029,0x2D2D775A,0x0F0F111E,0xB0B0CB7B,0x5454FCA8,0xBBBBD66D,0x16163A2C,
    };

    const Bit32 DecryptTable[256] =
    {
        0xF4A75051,0x4165537E,0x17A4C31A,0x275E963A,0xAB6BCB3B,0x9D45F11F,0xFA58ABAC,0xE303934B,
        0x30FA5520,0x766DF6AD,0xCC769188,0x024C25F5,0xE5D7FC4F,0x2ACBD7C5,0x35448026,0x62A38FB5,
        0xB15A49DE,0xBA1B6725,0xEA0E9845,0xFEC0E15D,0x2F7502C3,0x4CF01281,0x4697A38D,0xD3F9C66B,
        0x8F5FE703,0x929C9515,0x6D7AEBBF,0x5259DA95,0xBE832DD4,0x7421D358,0xE0692949,0xC9C8448E,
        0xC2896A75,0x8E7978F4,0x583E6B99,0xB971DD27,0xE14FB6BE,0x88AD17F0,0x20AC66C9,0xCE3AB47D,
        0xDF4A1863,0x1A3182E5,0x51336097,0x537F4562,0x6477E0B1,0x6BAE84BB,0x81A01CFE,0x082B94F9,
        0x48685870,0x45FD198F,0xDE6C8794,0x7BF8B752,0x73D323AB,0x4B02E272,0x1F8F57E3,0x55AB2A66,
        0xEB2807B2,0xB5C2032F,0xC57B9A86,0x3708A5D3,0x2887F230,0xBFA5B223,0x036ABA02,0x16825CED,
        0xCF1C2B8A,0x79B492A7,0x07F2F0F3,0x69E2A14E,0xDAF4CD65,0x05BED506,0x34621FD1,0xA6FE8AC4,
        0x2E539D34,0xF355A0A2,0x8AE13205,0xF6EB75A4,0x83EC390B,0x60EFAA40,0x719F065E,0x6E1051BD,
        0x218AF93E,0xDD063D96,0x3E05AEDD,0xE6BD464D,0x548DB591,0xC45D0571,0x06D46F04,0x5015FF60,
        0x98FB2419,0xBDE997D6,0x4043CC89,0xD99E7767,0xE842BDB0,0x898B8807,0x195B38E7,0xC8EEDB79,
        0x7C0A47A1,0x420FE97C,0x841EC9F8,0x00000000,0x80868309,0x2BED4832,0x1170AC1E,0x5A724E6C,
        0x0EFFFBFD,0x8538560F,0xAED51E3D,0x2D392736,0x0FD9640A,0x5CA62168,0x5B54D19B,0x362E3A24,
        0x0A67B10C,0x57E70F93,0xEE96D2B4,0x9B919E1B,0xC0C54F80,0xDC20A261,0x774B695A,0x121A161C,
        0x93BA0AE2,0xA02AE5C0,0x22E0433C,0x1B171D12,0x090D0B0E,0x8BC7ADF2,0xB6A8B92D,0x1EA9C814,
        0xF1198557,0x75074CAF,0x99DDBBEE,0x7F60FDA3,0x01269FF7,0x72F5BC5C,0x663BC544,0xFB7E345B,
        0x4329768B,0x23C6DCCB,0xEDFC68B6,0xE4F163B8,0x31DCCAD7,0x63851042,0x97224013,0xC6112084,
        0x4A247D85,0xBB3DF8D2,0xF93211AE,0x29A16DC7,0x9E2F4B1D,0xB230F3DC,0x8652EC0D,0xC1E3D077,
        0xB3166C2B,0x70B999A9,0x9448FA11,0xE9642247,0xFC8CC4A8,0xF03F1AA0,0x7D2CD856,0x3390EF22,
        0x494EC787,0x38D1C1D9,0xCAA2FE8C,0xD40B3698,0xF581CFA6,0x7ADE28A5,0xB78E26DA,0xADBFA43F,
        0x3A9DE42C,0x78920D50,0x5FCC9B6A,0x7E466254,0x8D13C2F6,0xD8B8E890,0x39F75E2E,0xC3AFF582,
        0x5D80BE9F,0xD0937C69,0xD52DA96F,0x2512B3CF,0xAC993BC8,0x187DA710,0x9C636EE8,0x3BBB7BDB,
        0x267809CD,0x5918F46E,0x9AB701EC,0x4F9AA883,0x956E65E6,0xFFE67EAA,0xBCCF0821,0x15E8E6EF,
        0xE79BD9BA,0x6F36CE4A,0x9F09D4EA,0xB07CD629,0xA4B2AF31,0x3F23312A,0xA59430C6,0xA266C035,
        0x4EBC3774,0x82CAA6FC,0x90D0B0E0,0xA7D81533,0x04984AF1,0xECDAF741,0xCD500E7F,0x91F62F17,
        0x4DD68D76,0xEFB04D43,0xAA4D54CC,0x9604DFE4,0xD1B5E39E,0x6A881B4C,0x2C1FB8C1,0x65517F46,
        0x5EEA049D,0x8C355D01,0x877473FA,0x0B412EFB,0x671D5AB3,0xDBD25292,0x105633E9,0xD647136D,
        0xD7618C9A,0xA10C7A37,0xF8148E59,0x133C89EB,0xA927EECE,0x61C935B7,0x1CE5EDE1,0x47B13C7A,
        0xD2DF599C,0xF2733F55,0x14CE7918,0xC737BF73,0xF7CDEA53,0xFDAA5B5F,0x3D6F14DF,0x44DB8678,
        0xAFF381CA,0x68C43EB9,0x24342C38,0xA3405FC2,0x1DC37216,0xE2250CBC,0x3C498B28,0x0D9541FF,
        0xA8017139,0x0CB3DE08,0xB4E49CD8,0x56C19064,0xCB84617B,0x32B670D5,0x6C5C7448,0xB85742D0,
    };

#elif defined(NN_BUILD_CONFIG_ENDIAN_LITTLE)
    // リトルエンディアン
    const int AesWordByte0 = 0UL;
    const int AesWordByte1 = 8UL;
    const int AesWordByte2 = 16UL;
    const int AesWordByte3 = 24UL;

    const int MixShift = 24UL;

    const Bit32 EncryptTable[256] =
    {
        0xA56363C6,0x847C7CF8,0x997777EE,0x8D7B7BF6,0x0DF2F2FF,0xBD6B6BD6,0xB16F6FDE,0x54C5C591,
        0x50303060,0x03010102,0xA96767CE,0x7D2B2B56,0x19FEFEE7,0x62D7D7B5,0xE6ABAB4D,0x9A7676EC,
        0x45CACA8F,0x9D82821F,0x40C9C989,0x877D7DFA,0x15FAFAEF,0xEB5959B2,0xC947478E,0x0BF0F0FB,
        0xECADAD41,0x67D4D4B3,0xFDA2A25F,0xEAAFAF45,0xBF9C9C23,0xF7A4A453,0x967272E4,0x5BC0C09B,
        0xC2B7B775,0x1CFDFDE1,0xAE93933D,0x6A26264C,0x5A36366C,0x413F3F7E,0x02F7F7F5,0x4FCCCC83,
        0x5C343468,0xF4A5A551,0x34E5E5D1,0x08F1F1F9,0x937171E2,0x73D8D8AB,0x53313162,0x3F15152A,
        0x0C040408,0x52C7C795,0x65232346,0x5EC3C39D,0x28181830,0xA1969637,0x0F05050A,0xB59A9A2F,
        0x0907070E,0x36121224,0x9B80801B,0x3DE2E2DF,0x26EBEBCD,0x6927274E,0xCDB2B27F,0x9F7575EA,
        0x1B090912,0x9E83831D,0x742C2C58,0x2E1A1A34,0x2D1B1B36,0xB26E6EDC,0xEE5A5AB4,0xFBA0A05B,
        0xF65252A4,0x4D3B3B76,0x61D6D6B7,0xCEB3B37D,0x7B292952,0x3EE3E3DD,0x712F2F5E,0x97848413,
        0xF55353A6,0x68D1D1B9,0x00000000,0x2CEDEDC1,0x60202040,0x1FFCFCE3,0xC8B1B179,0xED5B5BB6,
        0xBE6A6AD4,0x46CBCB8D,0xD9BEBE67,0x4B393972,0xDE4A4A94,0xD44C4C98,0xE85858B0,0x4ACFCF85,
        0x6BD0D0BB,0x2AEFEFC5,0xE5AAAA4F,0x16FBFBED,0xC5434386,0xD74D4D9A,0x55333366,0x94858511,
        0xCF45458A,0x10F9F9E9,0x06020204,0x817F7FFE,0xF05050A0,0x443C3C78,0xBA9F9F25,0xE3A8A84B,
        0xF35151A2,0xFEA3A35D,0xC0404080,0x8A8F8F05,0xAD92923F,0xBC9D9D21,0x48383870,0x04F5F5F1,
        0xDFBCBC63,0xC1B6B677,0x75DADAAF,0x63212142,0x30101020,0x1AFFFFE5,0x0EF3F3FD,0x6DD2D2BF,
        0x4CCDCD81,0x140C0C18,0x35131326,0x2FECECC3,0xE15F5FBE,0xA2979735,0xCC444488,0x3917172E,
        0x57C4C493,0xF2A7A755,0x827E7EFC,0x473D3D7A,0xAC6464C8,0xE75D5DBA,0x2B191932,0x957373E6,
        0xA06060C0,0x98818119,0xD14F4F9E,0x7FDCDCA3,0x66222244,0x7E2A2A54,0xAB90903B,0x8388880B,
        0xCA46468C,0x29EEEEC7,0xD3B8B86B,0x3C141428,0x79DEDEA7,0xE25E5EBC,0x1D0B0B16,0x76DBDBAD,
        0x3BE0E0DB,0x56323264,0x4E3A3A74,0x1E0A0A14,0xDB494992,0x0A06060C,0x6C242448,0xE45C5CB8,
        0x5DC2C29F,0x6ED3D3BD,0xEFACAC43,0xA66262C4,0xA8919139,0xA4959531,0x37E4E4D3,0x8B7979F2,
        0x32E7E7D5,0x43C8C88B,0x5937376E,0xB76D6DDA,0x8C8D8D01,0x64D5D5B1,0xD24E4E9C,0xE0A9A949,
        0xB46C6CD8,0xFA5656AC,0x07F4F4F3,0x25EAEACF,0xAF6565CA,0x8E7A7AF4,0xE9AEAE47,0x18080810,
        0xD5BABA6F,0x887878F0,0x6F25254A,0x722E2E5C,0x241C1C38,0xF1A6A657,0xC7B4B473,0x51C6C697,
        0x23E8E8CB,0x7CDDDDA1,0x9C7474E8,0x211F1F3E,0xDD4B4B96,0xDCBDBD61,0x868B8B0D,0x858A8A0F,
        0x907070E0,0x423E3E7C,0xC4B5B571,0xAA6666CC,0xD8484890,0x05030306,0x01F6F6F7,0x120E0E1C,
        0xA36161C2,0x5F35356A,0xF95757AE,0xD0B9B969,0x91868617,0x58C1C199,0x271D1D3A,0xB99E9E27,
        0x38E1E1D9,0x13F8F8EB,0xB398982B,0x33111122,0xBB6969D2,0x70D9D9A9,0x898E8E07,0xA7949433,
        0xB69B9B2D,0x221E1E3C,0x92878715,0x20E9E9C9,0x49CECE87,0xFF5555AA,0x78282850,0x7ADFDFA5,
        0x8F8C8C03,0xF8A1A159,0x80898909,0x170D0D1A,0xDABFBF65,0x31E6E6D7,0xC6424284,0xB86868D0,
        0xC3414182,0xB0999929,0x772D2D5A,0x110F0F1E,0xCBB0B07B,0xFC5454A8,0xD6BBBB6D,0x3A16162C,
    };

    const Bit32 DecryptTable[256] =
    {
        0x50A7F451,0x5365417E,0xC3A4171A,0x965E273A,0xCB6BAB3B,0xF1459D1F,0xAB58FAAC,0x9303E34B,
        0x55FA3020,0xF66D76AD,0x9176CC88,0x254C02F5,0xFCD7E54F,0xD7CB2AC5,0x80443526,0x8FA362B5,
        0x495AB1DE,0x671BBA25,0x980EEA45,0xE1C0FE5D,0x02752FC3,0x12F04C81,0xA397468D,0xC6F9D36B,
        0xE75F8F03,0x959C9215,0xEB7A6DBF,0xDA595295,0x2D83BED4,0xD3217458,0x2969E049,0x44C8C98E,
        0x6A89C275,0x78798EF4,0x6B3E5899,0xDD71B927,0xB64FE1BE,0x17AD88F0,0x66AC20C9,0xB43ACE7D,
        0x184ADF63,0x82311AE5,0x60335197,0x457F5362,0xE07764B1,0x84AE6BBB,0x1CA081FE,0x942B08F9,
        0x58684870,0x19FD458F,0x876CDE94,0xB7F87B52,0x23D373AB,0xE2024B72,0x578F1FE3,0x2AAB5566,
        0x0728EBB2,0x03C2B52F,0x9A7BC586,0xA50837D3,0xF2872830,0xB2A5BF23,0xBA6A0302,0x5C8216ED,
        0x2B1CCF8A,0x92B479A7,0xF0F207F3,0xA1E2694E,0xCDF4DA65,0xD5BE0506,0x1F6234D1,0x8AFEA6C4,
        0x9D532E34,0xA055F3A2,0x32E18A05,0x75EBF6A4,0x39EC830B,0xAAEF6040,0x069F715E,0x51106EBD,
        0xF98A213E,0x3D06DD96,0xAE053EDD,0x46BDE64D,0xB58D5491,0x055DC471,0x6FD40604,0xFF155060,
        0x24FB9819,0x97E9BDD6,0xCC434089,0x779ED967,0xBD42E8B0,0x888B8907,0x385B19E7,0xDBEEC879,
        0x470A7CA1,0xE90F427C,0xC91E84F8,0x00000000,0x83868009,0x48ED2B32,0xAC70111E,0x4E725A6C,
        0xFBFF0EFD,0x5638850F,0x1ED5AE3D,0x27392D36,0x64D90F0A,0x21A65C68,0xD1545B9B,0x3A2E3624,
        0xB1670A0C,0x0FE75793,0xD296EEB4,0x9E919B1B,0x4FC5C080,0xA220DC61,0x694B775A,0x161A121C,
        0x0ABA93E2,0xE52AA0C0,0x43E0223C,0x1D171B12,0x0B0D090E,0xADC78BF2,0xB9A8B62D,0xC8A91E14,
        0x8519F157,0x4C0775AF,0xBBDD99EE,0xFD607FA3,0x9F2601F7,0xBCF5725C,0xC53B6644,0x347EFB5B,
        0x7629438B,0xDCC623CB,0x68FCEDB6,0x63F1E4B8,0xCADC31D7,0x10856342,0x40229713,0x2011C684,
        0x7D244A85,0xF83DBBD2,0x1132F9AE,0x6DA129C7,0x4B2F9E1D,0xF330B2DC,0xEC52860D,0xD0E3C177,
        0x6C16B32B,0x99B970A9,0xFA489411,0x2264E947,0xC48CFCA8,0x1A3FF0A0,0xD82C7D56,0xEF903322,
        0xC74E4987,0xC1D138D9,0xFEA2CA8C,0x360BD498,0xCF81F5A6,0x28DE7AA5,0x268EB7DA,0xA4BFAD3F,
        0xE49D3A2C,0x0D927850,0x9BCC5F6A,0x62467E54,0xC2138DF6,0xE8B8D890,0x5EF7392E,0xF5AFC382,
        0xBE805D9F,0x7C93D069,0xA92DD56F,0xB31225CF,0x3B99ACC8,0xA77D1810,0x6E639CE8,0x7BBB3BDB,
        0x097826CD,0xF418596E,0x01B79AEC,0xA89A4F83,0x656E95E6,0x7EE6FFAA,0x08CFBC21,0xE6E815EF,
        0xD99BE7BA,0xCE366F4A,0xD4099FEA,0xD67CB029,0xAFB2A431,0x31233F2A,0x3094A5C6,0xC066A235,
        0x37BC4E74,0xA6CA82FC,0xB0D090E0,0x15D8A733,0x4A9804F1,0xF7DAEC41,0x0E50CD7F,0x2FF69117,
        0x8DD64D76,0x4DB0EF43,0x544DAACC,0xDF0496E4,0xE3B5D19E,0x1B886A4C,0xB81F2CC1,0x7F516546,
        0x04EA5E9D,0x5D358C01,0x737487FA,0x2E410BFB,0x5A1D67B3,0x52D2DB92,0x335610E9,0x1347D66D,
        0x8C61D79A,0x7A0CA137,0x8E14F859,0x893C13EB,0xEE27A9CE,0x35C961B7,0xEDE51CE1,0x3CB1477A,
        0x59DFD29C,0x3F73F255,0x79CE1418,0xBF37C773,0xEACDF753,0x5BAAFD5F,0x146F3DDF,0x86DB4478,
        0x81F3AFCA,0x3EC468B9,0x2C342438,0x5F40A3C2,0x72C31D16,0x0C25E2BC,0x8B493C28,0x41950DFF,
        0x7101A839,0xDEB30C08,0x9CE4B4D8,0x90C15664,0x6184CB7B,0x70B632D5,0x745C6C48,0x4257B8D0,
    };
#else
#error unknown NN_BUILD_CONFIG_ENDIAN
#endif

    inline Bit32 RotateLeft(Bit32 value, int shift) NN_NOEXCEPT
    {
        if (shift != 0)
        {
            return (value << shift) | (value >> (32 - shift));
        }
        else
        {
            return value;
        }
    }

#if 0
    inline void MixColumns(Bit32* pDst, const Bit32* pSrc) NN_NOEXCEPT
    {
        for (int i = 0; i < BlockWords; ++i)
        {
            Bit32 w = pSrc[i];
            Bit32 x = RotateLeft(w, 8UL) ^ RotateLeft(w, 16UL) ^ RotateLeft(w, 24UL);
            Bit32 c = (w & 0x80808080UL);
            Bit32 t = (((w & ~c) << 1UL) ^ ((c >> 7UL) * 0x1BUL));
            pDst[i]  = (x ^ t ^ RotateLeft(t, MixShift));
        }
    }
#endif

    inline void InvMixColumns(Bit32* pDst, const Bit32* pSrc) NN_NOEXCEPT
    {
        for (int i = 0; i < BlockWords; ++i)
        {
            const Bit32 x0 = pSrc[i];
            const Bit32 x1 = (((x0 & ~0x80808080UL) << 1UL) ^ (((x0 & 0x80808080UL) >> 7UL) * 0x1BUL));
            const Bit32 x2 = (((x1 & ~0x80808080UL) << 1UL) ^ (((x1 & 0x80808080UL) >> 7UL) * 0x1BUL));
            const Bit32 x3 = (((x2 & ~0x80808080UL) << 1UL) ^ (((x2 & 0x80808080UL) >> 7UL) * 0x1BUL));

            Bit32 w = static_cast<Bit32>(x0 ^ x3);
            w ^= (RotateLeft(w, MixShift) ^ x2);
            w ^= (RotateLeft(w, MixShift) ^ x1);
            w ^= (RotateLeft(w, MixShift) ^ x0);
            pDst[i] = w;
        }
    }

    inline Bit32 Func0(Bit32 x) NN_NOEXCEPT
    {
        return ( (SubBytesTable[(x >> AesWordByte0) & 0xFFUL] << AesWordByte3)
                 ^ (SubBytesTable[(x >> AesWordByte1) & 0xFFUL] << AesWordByte0)
                 ^ (SubBytesTable[(x >> AesWordByte2) & 0xFFUL] << AesWordByte1)
                 ^ (SubBytesTable[(x >> AesWordByte3) & 0xFFUL] << AesWordByte2) );
    }
    inline Bit32 Func1(Bit32 x) NN_NOEXCEPT
    {
        return ( (SubBytesTable[(x >> AesWordByte0) & 0xFFUL] << AesWordByte0)
                 ^ (SubBytesTable[(x >> AesWordByte1) & 0xFFUL] << AesWordByte1)
                 ^ (SubBytesTable[(x >> AesWordByte2) & 0xFFUL] << AesWordByte2)
                 ^ (SubBytesTable[(x >> AesWordByte3) & 0xFFUL] << AesWordByte3) );
    }
    inline Bit32 Func2(Bit32 x0, Bit32 x1, Bit32 x2, Bit32 x3) NN_NOEXCEPT
    {
        return ( RotateLeft(EncryptTable[(x0 >> AesWordByte0) & 0xFFUL], AesWordByte0)
                 ^ RotateLeft(EncryptTable[(x1 >> AesWordByte1) & 0xFFUL], AesWordByte1)
                 ^ RotateLeft(EncryptTable[(x2 >> AesWordByte2) & 0xFFUL], AesWordByte2)
                 ^ RotateLeft(EncryptTable[(x3 >> AesWordByte3) & 0xFFUL], AesWordByte3) );
    }
    inline Bit32 Func3(Bit32 x0, Bit32 x1, Bit32 x2, Bit32 x3) NN_NOEXCEPT
    {
        return ( (SubBytesTable[(x0 >> AesWordByte0) & 0xFFUL] << AesWordByte0)
                 | (SubBytesTable[(x1 >> AesWordByte1) & 0xFFUL] << AesWordByte1)
                 | (SubBytesTable[(x2 >> AesWordByte2) & 0xFFUL] << AesWordByte2)
                 | (SubBytesTable[(x3 >> AesWordByte3) & 0xFFUL] << AesWordByte3) );

    }
    inline Bit32 Func4(Bit32 x0, Bit32 x1, Bit32 x2, Bit32 x3) NN_NOEXCEPT
    {
        return ( RotateLeft(DecryptTable[(x0 >> AesWordByte0) & 0xFFUL], AesWordByte0)
                 ^ RotateLeft(DecryptTable[(x1 >> AesWordByte1) & 0xFFUL], AesWordByte1)
                 ^ RotateLeft(DecryptTable[(x2 >> AesWordByte2) & 0xFFUL], AesWordByte2)
                 ^ RotateLeft(DecryptTable[(x3 >> AesWordByte3) & 0xFFUL], AesWordByte3) );
    }
    inline Bit32 Func5(Bit32 x0, Bit32 x1, Bit32 x2, Bit32 x3) NN_NOEXCEPT
    {
        return ( (InvSubBytesTable[(x0 >> AesWordByte0) & 0xFFUL] << AesWordByte0)
                 ^ (InvSubBytesTable[(x1 >> AesWordByte1) & 0xFFUL] << AesWordByte1)
                 ^ (InvSubBytesTable[(x2 >> AesWordByte2) & 0xFFUL] << AesWordByte2)
                 ^ (InvSubBytesTable[(x3 >> AesWordByte3) & 0xFFUL] << AesWordByte3) );

    }

}   // anonymous namespace


template <size_t KeySize>
AesImpl<KeySize>::~AesImpl() NN_NOEXCEPT
{
    ClearMemory(this, sizeof(*this));
}

template <size_t KeySize>
void AesImpl<KeySize>::Initialize(const void* pKey, size_t keySize, bool isEncryptionKey) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(keySize == KeySize, "invalid key size. keySize(=%d) must be either 16, 24, 32", keySize);

    /* 鍵の拡張 */
    const int KeySizeInWord = static_cast<int>(keySize / sizeof(Bit32));
    Bit32*    pDst = m_RoundKey;
    Bit32     reg;

    /* 初期鍵をコピー */
    std::memcpy(pDst, pKey, keySize);

    /* 前段の最後のwordを使用 */
    reg = pDst[KeySizeInWord - 1];

    for (int i = KeySizeInWord; i < (RoundCount + 1) * 4; ++i)
    {
        /* SubWord + RotWord + XorRcon */
        if ((i % KeySizeInWord) == 0)
        {
            reg = Func0(reg);
            reg ^= (RoundKeyRcon0[i / KeySizeInWord - 1] << AesWordByte0);
        }
        else if ((KeySizeInWord > 6) && ((i % KeySizeInWord) == 4))
        {
            reg = Func1(reg);
        }

        reg ^= pDst[i - KeySizeInWord];
        pDst[i] = reg;
    }

    /*
     * InvMixColumns(a ^ b) = InvMixColumns(a) ^ InvMixColumns(b)
     * であることを利用して AddRoundKey 処理を遅延し,
     * 復号化でも暗号化と同様にテーブル検索を効率化する.
     */
    if (!isEncryptionKey)
    {
        for (int i = 1; i < RoundCount; ++i)
        {
            InvMixColumns(&m_RoundKey[BlockWords * i], &m_RoundKey[BlockWords * i]);
        }
    }
}

template <size_t KeySize>
void AesImpl<KeySize>::EncryptBlock(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) const NN_NOEXCEPT
{
    const Bit32* pKey32 = m_RoundKey;
    const Bit32* pSrc32 = static_cast<const Bit32*>(pSrc);
    Bit32* pDst32 = static_cast<Bit32*>(pDst);
    int round = RoundCount;

    NN_SDK_REQUIRES(dstSize == BlockSize && srcSize == BlockSize, "invalid block size is specified");
    NN_UNUSED(dstSize);
    NN_UNUSED(srcSize);

    /* AddRoundKey */
    Bit32 tmp0 = pSrc32[0] ^ pKey32[0];
    Bit32 tmp1 = pSrc32[1] ^ pKey32[1];
    Bit32 tmp2 = pSrc32[2] ^ pKey32[2];
    Bit32 tmp3 = pSrc32[3] ^ pKey32[3];
    pKey32 += 4;

    while (--round > 0)
    {
        /* ShiftRows + SubBytes + MixColumn */
        const Bit32 mix0 = Func2(tmp0, tmp1, tmp2, tmp3);
        const Bit32 mix1 = Func2(tmp1, tmp2, tmp3, tmp0);
        const Bit32 mix2 = Func2(tmp2, tmp3, tmp0, tmp1);
        const Bit32 mix3 = Func2(tmp3, tmp0, tmp1, tmp2);

        /* AddRoundKey */
        tmp0 = mix0 ^ pKey32[0];
        tmp1 = mix1 ^ pKey32[1];
        tmp2 = mix2 ^ pKey32[2];
        tmp3 = mix3 ^ pKey32[3];
        pKey32 += 4;
    }

    /* ShiftRows + SubBytes + AddRoundKey */
    pDst32[0] = (pKey32[0] ^ Func3(tmp0, tmp1, tmp2, tmp3));
    pDst32[1] = (pKey32[1] ^ Func3(tmp1, tmp2, tmp3, tmp0));
    pDst32[2] = (pKey32[2] ^ Func3(tmp2, tmp3, tmp0, tmp1));
    pDst32[3] = (pKey32[3] ^ Func3(tmp3, tmp0, tmp1, tmp2));
}

template <size_t KeySize>
void AesImpl<KeySize>::DecryptBlock(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) const NN_NOEXCEPT
{
    const Bit32* pKey32 = m_RoundKey;
    const Bit32* pSrc32 = static_cast<const Bit32*>(pSrc);
    Bit32* pDst32 = static_cast<Bit32*>(pDst);
    int round = RoundCount;

    pKey32 += 4 * round;

    NN_SDK_REQUIRES(dstSize == BlockSize && srcSize == BlockSize, "invalid block size is specified");
    NN_UNUSED(dstSize);
    NN_UNUSED(srcSize);

    /* AddRoundKey */
    Bit32 tmp0 = pSrc32[0] ^ pKey32[0];
    Bit32 tmp1 = pSrc32[1] ^ pKey32[1];
    Bit32 tmp2 = pSrc32[2] ^ pKey32[2];
    Bit32 tmp3 = pSrc32[3] ^ pKey32[3];
    pKey32 -= 4;

    /*
     * InvMixColumns(a ^ b) = InvMixColumns(a) ^ InvMixColumns(b)
     * であることを利用して実際の AddRoundKey 処理を遅延し,
     * 暗号化と同様の形式でテーブル検索を効率化する.
     */
    while (--round > 0)
    {
        /* InvShiftRows + InvSubBytes + InvMixColumns(except pKey) */
        const Bit32 mix0 = Func4(tmp0, tmp3, tmp2, tmp1);
        const Bit32 mix1 = Func4(tmp1, tmp0, tmp3, tmp2);
        const Bit32 mix2 = Func4(tmp2, tmp1, tmp0, tmp3);
        const Bit32 mix3 = Func4(tmp3, tmp2, tmp1, tmp0);

        /* AddRoundKey(with InvMixColumns) */
        tmp0 = mix0 ^ pKey32[0];
        tmp1 = mix1 ^ pKey32[1];
        tmp2 = mix2 ^ pKey32[2];
        tmp3 = mix3 ^ pKey32[3];
        pKey32 -= 4;
    }

    /* InvShiftRows + InvSubBytes + AddRoundKey */
    pDst32[0] = (pKey32[0] ^ Func5(tmp0, tmp3, tmp2, tmp1));
    pDst32[1] = (pKey32[1] ^ Func5(tmp1, tmp0, tmp3, tmp2));
    pDst32[2] = (pKey32[2] ^ Func5(tmp2, tmp1, tmp0, tmp3));
    pDst32[3] = (pKey32[3] ^ Func5(tmp3, tmp2, tmp1, tmp0));
}

/* テンプレートの明示的実体化 */
template class AesImpl<16>;
template class AesImpl<24>;
template class AesImpl<32>;

}}}

