﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#ifndef __cplusplus

//----------------------------------------------------------------------------
//  全てのライブラリで共通して使用するマクロの定義（oasiscafe_clang 用）
//----------------------------------------------------------------------------

#include <nn/TargetConfigs/build_Compiler.h>

#if !defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_OASIS_CAFE_CLANG)
    #error  "OasisCafe CLANG コンパイラーが指定されていません。"
#endif


//----------------------------------------------------------------------------
//  言語サポート
//----------------------------------------------------------------------------

//  アライメントのバイトサイズを指定します。
#define NNC_ALIGNAS( alignment ) __attribute__((aligned(alignment)))

//  アライメントのバイトサイズを取得します。
#define NNC_ALIGNOF( type ) __alignof__( type )

//  決して返らない関数に対するコンパイラ警告を抑制します。
#define NNC_NORETURN __attribute__((noreturn))

//  関数が例外を投げないことを宣言します。
// C++ 専用機能
//#define NNC_NOEXCEPT throw()

//  関数が仮想関数をオーバーライドすることを宣言します。
// C++ 専用機能
//#define NNC_OVERRIDE

//  型変換演算子が暗黙的に適用されることを禁止します。
// C++ 専用機能
//#define NNC_EXPLICIT_OPERATOR operator

//  インライン化を意図していないことを明示します。
#define NNC_NOINLINE __attribute__((noinline))

//  インライン化を強制する意図を明示します。
#define NNC_FORCEINLINE __attribute__((always_inline)) inline

//  インライン化する意図を明示します。
#define NNC_INLINE inline

//  現在の関数の名前です。
#define NNC_CURRENT_FUNCTION_NAME __FUNCTION__

#define NNC_WEAK_SYMBOL          __attribute__((weak))

//----------------------------------------------------------------------------
//  環境毎の問題回避
//----------------------------------------------------------------------------

//  宣言時に値を指定したクラス定数の定義におけるエラーを回避します。
// C++ 専用機能
//#define NNC_DEFINE_STATIC_CONSTANT( definition ) definition

//  外部の関数型マクロによるシンボル置き換えを抑止します。
#define NNC_PREVENT_MACRO_FUNC


//----------------------------------------------------------------------------
//  ユーティリティ
//----------------------------------------------------------------------------

//  コピーコンストラクタとコピー代入演算子を禁止します。
// C++ 専用機能
//#define NNC_DISALLOW_COPY( type ) \
//private: \
//    type( const type& ); \
//    type& operator=( const type& )

//  ムーブコンストラクタとムーブ代入演算子を禁止します。
// C++ 専用機能
//#define NNC_DISALLOW_MOVE( type ) \
//private: \
//    type( type&& ); \
//    type& operator=( type&& )

//  型または変数のビットサイズを取得します。
//  ヘッダのインクルードを不要にするため CHAR_BIT の値を展開して記述しています。
//  CHAR_BIT が異なる環境に対応する際は切り分けを行います。
#define NNC_BITSIZEOF( type ) ( (int)( sizeof( type ) ) * 8 )

//  文字列リテラルを作成します。
#define NNC_MACRO_STRINGIZE( text ) NNC_MACRO_STRINGIZE_IMPL_( text )

//  シンボルを連結します。
#define NNC_MACRO_CONCATENATE( x, y ) NNC_MACRO_CONCATENATE_IMPL_( x, y )

//  内部実装用
#define NNC_MACRO_STRINGIZE_IMPL_( text ) #text
#define NNC_MACRO_CONCATENATE_IMPL_( x, y ) x##y

//  重複しない一時シンボルを作成します。
#define NNC_MAKE_TEMPORARY_NAME( prefix ) NNC_MACRO_CONCATENATE( prefix, __COUNTER__ )

//  switch 文において、case ラベルの処理が直後の case ラベルに
//  フォールスルーすることを明示します。
//  C言語版では未対応
//#define NNC_FALL_THROUGH

//  switch 文において、default ラベルに決して到達しないことを表明します。
//  default ラベルに到達した場合、実行を停止します。
#define NNC_UNEXPECTED_DEFAULT \
    nndetailUnexpectedDefaultImpl( \
        NNC_CURRENT_FUNCTION_NAME, __FILE__, __LINE__)

//  内部実装用
NNC_NORETURN void nndetailUnexpectedDefaultImpl(
    const char* functionName,
    const char* fileName,
    int lineNumber);

// クラス内部に暗黙にパディングが挿入されたことの警告を有効にします。
#define NNC_ENABLE_WARNING_IMPLICIT_PADDING_WITHIN_CLASS \
    _Pragma("GCC diagnostic warning  \"-Wpadded\"")

// クラス内部に、1 バイトのサイズを持つ無名のメンバを定義します。このメンバは、1 バイトアラインです。
// Warning が出るため C言語版では未対応
//#define NNC_PADDING1 char :  8

//----------------------------------------------------------------------------
//  警告対策
//----------------------------------------------------------------------------

//  意図した暗黙的呼び出し可能な変換コンストラクタ、変換演算子に対する静的解析ツール警告を
//  抑制します。
// C++ 専用機能
//#define NNC_IMPLICIT

//  意図した未使用変数に対するコンパイラ警告を抑制します。
#define NNC_UNUSED( variable ) ( void )( variable )

//  意図した定数分岐を明示してコンパイラ警告を抑制します。
#define NNC_STATIC_CONDITION( condition ) ( ( void )0, ( condition ) )

//  コンパイラの警告レベルを退避します。
#define NNC_PRAGMA_PUSH_WARNINGS     _Pragma("GCC diagnostic push")

//  コンパイラの警告レベルを復帰します。
#define NNC_PRAGMA_POP_WARNINGS      _Pragma("GCC diagnostic pop")

//  定数のオーバーフローに対するコンパイラの警告を無効化します。
#define NNC_DISABLE_WARNING_OVERFLOW _Pragma("GCC diagnostic ignored  \"-Woverflow\"")

//  関数内で宣言された静的変数に対するコンパイラの警告を無効化します。
#define NNC_DISABLE_WARNING_LOCAL_STATIC

//  未使用変数に対するコンパイラの警告を無効化します。
#define NNC_DISABLE_WARNING_UNUSED_VAR   _Pragma("GCC diagnostic ignored  \"-Wunused-variable\"")


#endif
